# -*- coding: utf-8 -*-

import os
import logging
import shutil
import re

import sandbox.common.types.client as ctc

import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
import sandbox.sandboxsdk.ssh as sdk_ssh

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import parameters


class BuildClustermasterDeb(SandboxTask):
    type = 'BUILD_CLUSTERMASTER_DEB'

    client_tags = ctc.Tag.Group.LINUX

    class ResourceIds(parameters.SandboxStringParameter):
        name = "resource_ids"
        description = "Binary resources ids"

    class DebianFilesArcadiaUrl(parameters.SandboxStringParameter):
        name = "debian_files_arcadia_url"
        description = "Debian files arcadia url"
        default_value = "packages/debian/tools/yandex-search-cm"

    class Branch(parameters.SandboxStringParameter):
        name = "branch"
        description = "Branch"

    class Revision(parameters.SandboxStringParameter):
        name = "revision"
        description = "Revision number"

    class RepositoryId(parameters.SandboxStringParameter):
        name = "repository_id"
        description = "Repository ID"

    class ReleaseComments(parameters.SandboxStringParameter):
        name = "release_comments"
        description = "Release comments"
        multiline = True

    # see comments below to find more info
    class TestingPackage(parameters.SandboxBoolParameter):
        name = "testing_package"
        description = (
            "Build packages for testing."
            " It means that package is not uploaded to repo"
            " (right way is uploading to unstable repo but we don't have one yet)."
        )
        default_value = False

    class RebuildPackage(parameters.SandboxBoolParameter):
        name = "rebuild_package"
        description = "Rebuild package. Version should be the same. -n suffix is added to version."

    class RebuildPackageSuffix(parameters.SandboxStringParameter):
        name = "rebuild_package_suffix"
        description = "Rebuild package suffix"

    input_parameters = [
        ResourceIds, DebianFilesArcadiaUrl, Branch,
        Revision, RepositoryId, ReleaseComments,
        TestingPackage, RebuildPackage, RebuildPackageSuffix
    ]

    LOCAL_PACKAGE_DIR_PREFIX = 'package/cm-'

    DUPLOAD_CONF = {
        'search': {
            'fqdn': "search.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': "robot-admins",
            'incoming': "/repo/search/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        }
    }

    def on_execute(self):
        if not re.match('\d+', self.ctx['revision']):
            raise Exception("Wrong self.ctx['revision']: %s" % self.ctx['revision'])
        revision = self.ctx['revision']

        if not re.match('\d+', self.ctx['branch']):
            raise Exception("Wrong self.ctx['branch']: %s" % self.ctx['branch'])
        branch = self.ctx['branch']
        arcadia_branch = "trunk"

        # package dir
        local_package_dir = self.LOCAL_PACKAGE_DIR_PREFIX + revision
        package_dir = os.path.join(self.abs_path(), local_package_dir)
        make_folder(package_dir)
        run_process('chmod -R a+w %s' % package_dir, wait=True, check=True)
        logging.info('Package dir: %s' % package_dir)

        svn_url = 'arcadia:/arc/%s/arcadia/%s' % (arcadia_branch, self.ctx['debian_files_arcadia_url'])
        Arcadia.checkout(svn_url, os.path.join(package_dir, '.'))

        # copy binary
        if not self.ctx['resource_ids']:
            raise Exception('resource_ids should be specified')
        ids = self.ctx['resource_ids'].split(',')
        for resId in ids:
            self._copyBinaryResource(resId, local_package_dir)

        os.chdir(package_dir)

        epoch = '1:'  # epoch is used to leave old version scheme behind; to read about epoch type 'man deb-version'

        rebuild_package = self.ctx.get('rebuild_package', False)
        rebuild_package_suffix = self.ctx['rebuild_package_suffix']
        if rebuild_package and not rebuild_package_suffix:
            raise Exception("Rebuild package option is set buf rebuild package suffix is not provided")
        version = epoch + branch + '.' + revision + '.' + arcadia_branch + ('' if not rebuild_package else ('-' + rebuild_package_suffix))
        logging.info('Version is %s' % version)

        environment = os.environ.copy()
        environment['DEBEMAIL'] = 'Robot Admins <robot-admins@yandex-team.ru>'
        with sandbox.projects.common.gnupg.GpgKey(self, "ROBOT-ADMINS", "robot-admins-gpg-private", "robot-admins-gpg-public"):
            run_process('dch -v %s -D unstable --force-distribution --no-auto-nmu Package was created by sandbox task' % (version), wait=True, check=True, log_prefix='dch')
            run_process('debuild --no-tgz-check -krobot-admins@yandex-team.ru', wait=True, check=True, log_prefix='debuild')

            shutil.copytree(os.path.join(self.abs_path(), 'package'), os.path.join(self.log_path(), 'package'))

            # Simply don't upload package to repo if testing_package checker is selected - freebsd hack with adding '_testing'
            # suffix to name will not work here. Problem with hack is that we need to change with sed both control and changelog
            # files - this is too complicated (or disgusting :-). Also on debian it is simple to have two repos - stable/unstable.
            # And the last reason is that we have testing cluster running freebsd not linux.
            testing_package = self.ctx.get('testing_package', True)
            if not testing_package and self.ctx['repository_id']:
                with sandbox.projects.common.debpkg.DebRelease(self.DUPLOAD_CONF) as deb:
                    with sdk_ssh.Key(self, "ROBOT-ADMINS", "robot-admins-ssh"):
                        deb.debrelease(['--to', self.ctx['repository_id']])

    def _copyBinaryResource(self, res_id, local_package_dir):
        self.sync_resource(res_id)
        res = channel.sandbox.get_resource(res_id)
        resource_dir = os.path.join(self.abs_path(), local_package_dir)
        make_folder(resource_dir)
        shutil.copy(res.path, resource_dir)


__Task__ = BuildClustermasterDeb
