# -*- coding: utf-8 -*-

import os
import json
import re
import subprocess
import time

# stuff from sandbox
from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.common.types.task import Status
import sandbox.common.types.client as client

# Это имя будет сохранено в tar-архиве
from sandbox.projects.common.nanny import nanny

# auxiliarily gencfg stuff
from sandbox.projects.common.gencfg.task import IGencfgBuildTask, get_topology_mongo

ARCHIVE_GENERATOR_NAME = 'generator'

# Переиспользуемые поля контекста
SVN_COMMIT_KEY = 'git_commit'

WAIT_TIMEOUT = 60


class GeneratorGitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Svn tag'
    default_value = 'trunk'


class UseSeparateTaskToBuildConfigs(SandboxBoolParameter):
    name = 'separate_task_for_configs'
    description = 'Use separate task to build configs'
    default_value = True


class GenerateConfigsParameter(SandboxBoolParameter):
    name = 'generate_all'
    description = 'Run generator script'
    default_value = True


class GenerateDiff(SandboxBoolParameter):
    name = 'generate_diff_to_prev'
    description = 'Generate diff to previous tag'
    default_value = False


class MaxDiffSize(SandboxIntegerParameter):
    name = 'max_diff_size'
    description = 'If generate diff flag is true the total diff size muse be limited '
    default_value = 190039740


class IgnoreDiffLimit(SandboxBoolParameter):
    name = 'ignore_diff_limit'
    description = 'Ignore max diff size limit '
    default_value = False


class GenerateImproxyConfigs(SandboxBoolParameter):
    name = 'generate_improxy_configs'
    description = 'Generate improxy configs'
    default_value = False


class BuildConfigGenerator(nanny.ReleaseToNannyTask, IGencfgBuildTask):
    type = 'BUILD_CONFIG_GENERATOR'

    client_tags = client.Tag.CUSTOM_GENCFG_BUILD

    input_parameters = [
        GeneratorGitTagParameter,
        GenerateConfigsParameter,
        GenerateDiff,
        MaxDiffSize,
        IgnoreDiffLimit,
        GenerateImproxyConfigs,
        UseSeparateTaskToBuildConfigs,
    ]

    def arcadia_info(self):
        return self.ctx.get(SVN_COMMIT_KEY, None), 'gencfg/%s' % (self.ctx[GeneratorGitTagParameter.name]), None

    def tag_is_major(self):
        return re.match(r'^stable-\d+-r1$', self.ctx[GeneratorGitTagParameter.name]) is not None

    def get_last_full_commit(self):
        for tag in get_topology_mongo()['tags'].find(sort=[('commit', -1)]):
            if tag.get('fullstate', False):
                return tag['commit']

    def is_tag_built(self, tag_name):
        builded_tags = list(get_topology_mongo()['tags'].find({'tag': tag_name}))
        return len(builded_tags) > 0

    def get_nanny_oauth_token(self):
        return self.get_vault_data('GENCFG', 'gencfg_default_oauth')

    def on_execute(self):
        tag_name = self.ctx[GeneratorGitTagParameter.name]
        if self.is_tag_built(tag_name) and False:
            raise RuntimeError('Tag {} has already been built. Rebuilding tags is not allowed.'.format(tag_name))

        if self.ctx[UseSeparateTaskToBuildConfigs.name] and "build_configs_via_api_task" not in self.ctx:
            build_conigs_via_api = sdk2.Task["BUILD_CONFIGS_VIA_API"]
            if self.ctx[GenerateImproxyConfigs.name]:
                task = build_conigs_via_api(
                    build_conigs_via_api.current,
                    description="Build configs via api",
                    owner=self.owner,
                    priority=self.priority,
                    configs=[
                        # Approved by sereglond@, mvel@ and desertfury@
                        # 'l7',  Disable l7 configs build as L7 team tests it separately
                        'media-balancers',
                        'improxy',
                    ],
                ).enqueue()
            else:
                task = build_conigs_via_api(
                    build_conigs_via_api.current,
                    description="Build configs via api",
                    owner=self.owner,
                    priority=self.priority
                ).enqueue()
            self.ctx["build_configs_via_api_task"] = task.id

        self.clone_and_install_generator(
            tag=self.ctx[GeneratorGitTagParameter.name],
            precalc_caches=True,
            load_cache_resource=True,
        )

        if self.ctx[GenerateDiff.name]:
            generate_diff_cmd = self.generate_diff_create_cmd(
                self.get_gencfg_path(),
                ["all"],
                r"^stable-(\d+)-r(\d+)$",
                commit_filter="lambda x: 'backend' not in x.labels",
            )
            self.add_background_command(generate_diff_cmd, "diff_builder")

        extras = []
        if self.tag_is_major():
            extras += ['--full']
        self.add_background_command([
            '/skynet/python/bin/python',
            './utils/mongo/populate_searcher_lookup.py'
        ] + extras, 'populate_searcher_lookup')
        self.add_background_command([
            '/skynet/python/bin/python',
            './utils/mongo/populate_search_map.py'
        ] + extras, 'populate_search_map')

        import_to_gencfg_trunk_cmd = ['./utils/mongo/populate_gencfg_trunk.py']
        self.add_background_command(import_to_gencfg_trunk_cmd, 'import_to_gencfg_trunk')

        # /trunk/hosts/hosts_to_groups
        sync_hosts_info_with_mongo_cmd = [
            './utils/mongo/sync_hosts_info_with_mongo.py',
            '-s', 'hosts_to_groups',
            '--prod-run',
            '--auto-clean',
            '-v'
        ]
        self.add_background_command(sync_hosts_info_with_mongo_cmd, 'sync_hosts_info-hosts_to_groups')

        # /trunk/hosts/hosts_to_hardware
        sync_hosts_info_with_mongo_cmd = [
            './utils/mongo/sync_hosts_info_with_mongo.py',
            '-s', 'hosts_to_hardware',
            '--prod-run',
            '--auto-clean',
            '-v'
        ]
        self.add_background_command(sync_hosts_info_with_mongo_cmd, 'sync_hosts_info-hosts_to_hardware')

        # /trunk/slbs
        export_slbs_cmd = ['./utils/mongo/export_slbs.py']
        self.add_background_command(export_slbs_cmd, 'export_slbs')

        populate_staff_cmd = [
            './utils/mongo/populate_staff_cache.py',
            '--oauth-token', self.get_vault_data('GENCFG', 'gencfg_default_oauth')
        ]
        self.add_background_command(populate_staff_cmd, 'populate_staff_cache')

        self.add_background_command([
            '/skynet/python/bin/python',
            './utils/mongo/populate_instances_tags.py',
        ], 'populate_instances_tags')

        export_cpumodels_cmd = ['./utils/mongo/export_cpumodels.py']
        self.add_background_command(export_cpumodels_cmd, 'export_cpumodels')

        if self.ctx[UseSeparateTaskToBuildConfigs.name]:
            self.build_generator(use_separate_task_for_configs=True, create_resources=False)
            if "build_configs_via_api_task" in self.ctx:
                api = common.rest.Client()
                task_status = None
                timeout = 1
                while timeout < WAIT_TIMEOUT:
                    task_info = api.task[self.ctx["build_configs_via_api_task"]].read()
                    if task_info['status'] != task_status:
                        task_status = task_info['status']
                    if task_status in Status.Group.BREAK or task_status == Status.FAILURE:
                        raise Exception('Task stopped with status {}'.format(task_status))
                    elif task_status == 'SUCCESS':
                        break
                    time.sleep(timeout)
                    timeout += 1
                    if timeout == WAIT_TIMEOUT:
                        raise Exception('Child task timed out')

                resources = api.task[self.ctx["build_configs_via_api_task"]].resources.read()
                for resource in resources["items"]:
                    current_path = os.getcwd()
                    os.chdir(self.get_gencfg_path())
                    if resource["type"] == "CONFIGS_DIRECTORY":
                        subprocess.check_call(["sky", "get", "-wu", "sbr:{}".format(resource["id"])])
                    os.chdir(current_path)
            self.build_generator(create_resources=True, run_build=False)
        else:
            self.build_generator()

        if self.ctx[GenerateDiff.name]:
            self.generate_diff_process_result()
            diff_size = os.stat(self.get_nanny_file()).st_size
            if (not self.ctx[IgnoreDiffLimit.name]) and (diff_size > self.ctx[MaxDiffSize.name]):
                raise Exception('Diff is too big. Actual size: {}. Max size: {}'.format(diff_size, self.ctx[MaxDiffSize.name]))

        self.ctx[SVN_COMMIT_KEY] = self.get_last_commit(self.get_gencfg_path())

    def on_release(self, additional_parameters):
        release_status = additional_parameters.get('release_status')
        # release_subject = additional_parameters.get('release_subject')

        for resource in self.list_resources():
            if resource.type.name != 'TASK_LOGS':
                channel.sandbox.set_resource_attribute(resource.id, 'tag', self.ctx[GeneratorGitTagParameter.name])

        # send released resources to Nanny and add info about them to task's info field
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        if 'nannyfile_id' in self.ctx:
            nanny_diff_resource = self.sync_resource(self.ctx['nannyfile_id'])

            with open(nanny_diff_resource) as fp:
                nanny_diff = json.load(fp)
                nanny.ReleaseToNannyTask.on_gencfg_release(self, nanny_diff, additional_parameters)

        # announce chats
        if 'announce_id' in self.ctx:
            announce_resource = self.sync_resource(self.ctx['announce_id'])

            with open(announce_resource) as fp:
                announce_content = fp.read()
                if len(announce_content) > 0:
                    # currently have to clone gencfg here
                    gencfg_path = self.abs_path('release_gencfg_path')
                    self.clone_and_install_generator(
                        gencfg_path=gencfg_path,
                        tag=self.ctx[GeneratorGitTagParameter.name],
                        create_resource=False,
                        precalc_caches=True,
                        load_cache_resource=True,
                    )
                    cmd = ['./utils/common/manipulate_telegram.py', '-a', 'announce', '-m', announce_content]
                    run_process(cmd, work_dir=gencfg_path, log_prefix='announce_to_telechats', wait=True, check=True)

        # leave last 4 major tags in mongo
        # self.remove_old_tags(4) temporarily disabled
        self.mark_released_resources(release_status)
        self.retry_mark_tag_released(
            self.ctx[GeneratorGitTagParameter.name],
            self.ctx[SVN_COMMIT_KEY],
            self.tag_is_major(),
            self.get_last_full_commit()
        )

        self.tags = [self.ctx[GeneratorGitTagParameter.name]]

        # notify tasks and users about tag being released
        # self.report_mail_on_release(release_status, release_subject)
        # self.report_startrek_on_release(self.ctx[GeneratorGitTagParameter.name])

    def retry_mark_tag_released(self, tag, commit, full=False, diff_to=None):
        try:
            delays = (3, 6, 12, 24, 48, 96)
            for i, delay in enumerate(delays):
                try:
                    return self.mark_tag_released(
                        self.ctx[GeneratorGitTagParameter.name],
                        self.ctx[SVN_COMMIT_KEY],
                        self.tag_is_major(),
                        self.get_last_full_commit()
                    )
                except Exception as e:
                    print('{} [{}]: {}'.format(i, type(e), e))
                    if i + 1 >= len(delays):
                        raise
                    time.sleep(delay)
        except Exception:
            print('Mark task resources as NOT_RELEASED')
            self.mark_released_resources(False)
            raise


__Task__ = BuildConfigGenerator
