# -*- coding: utf-8 -*-

import os
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter, SandboxBoolParameter
import sandbox.sandboxsdk.process as sdk_process
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.gencfg.task import IGencfgTask

REPOS_SRC = [
    {
        "name": "full",
        "addr": "https://git.qe-infra.yandex-team.ru/scm/nanny/gencfg.git",
        "full_postactions": [
            ["/usr/bin/env", "git", "remote", "set-url", "--push", "origin", "ssh://git@git.qe-infra.yandex-team.ru/nanny/gencfg.git"],
            ["./checkout.sh"],
            ["./install.sh", "tests"],
            ["./utils/common/precalc_caches.py"],
            ["rm", "-rf", "venv"],  # we do not want venv inside resource
            ["rm", "-rf", "custom_generators/balancer_gencfg/venv"],  # we do not want venv inside resource
        ],
        "incremental_postactions": [
            ["./pull.sh"],
            ["./install.sh", "tests"],
            ["./utils/common/precalc_caches.py"],
            ["rm", "-rf", "venv"],  # we do not want venv inside resource
            ["rm", "-rf", "custom_generators/balancer_gencfg/venv"],  # we do not want venv inside resource
        ],
    },
    {
        "name": "main",
        "addr": "https://git.qe-infra.yandex-team.ru/scm/nanny/gencfg.git",
        "full_postactions": [
            ["/usr/bin/env", "git", "remote", "set-url", "--push",  "origin", "ssh://git@git.qe-infra.yandex-team.ru/nanny/gencfg.git"],
        ],
    },
    {
        "name": "db",
        "addr": "https://git.qe-infra.yandex-team.ru/scm/nanny/gencfg.db.git",
        "full_postactions": [
            ["/usr/bin/env", "git", "remote", "set-url", "--push",  "origin", "ssh://git@git.qe-infra.yandex-team.ru/nanny/gencfg.db.git"],
        ],
    },
    {
        "name": "balancer",
        "addr": "https://git.qe-infra.yandex-team.ru/scm/nanny/balancer-gencfg.git",
        "full_postactions": [
            ["/usr/bin/env", "git", "remote", "set-url", "--push",  "origin", "ssh://git@git.qe-infra.yandex-team.ru/nanny/balancer-gencfg.git"],
        ],
    }
]

REPOS = {x["name"]: x for x in REPOS_SRC}


RESULT_DIR = "resource"
RESULT_TAR_FILE = "resource.tar.gz"


class ParamRepoType(SandboxStringParameter):
    name = 'repo_type'
    description = 'Repository Type'
    default_value = ""
    choices = [(x["name"], x["name"]) for x in REPOS_SRC]


class ParamIsIncremental(SandboxBoolParameter):
    name = 'is_incremental'
    description = 'Is Incremental'
    default_value = True


class ParamMajorVersion(SandboxIntegerParameter):
    name = 'major_version'
    description = 'Build Major Version'
    default_value = -1


class BuildConfigGeneratorRepo(IGencfgTask):
    type = 'BUILD_CONFIG_GENERATOR_REPO'

    input_parameters = [ParamRepoType, ParamIsIncremental, ParamMajorVersion]

    def on_enqueue(self):
        if self.ctx[ParamRepoType.name] not in REPOS:
            raise Exception("Unsupported repo type \"%s\"" % self.ctx[ParamRepoType.name])

    def on_execute(self):
        repo_type = self.ctx[ParamRepoType.name]
        repo_opts = REPOS[repo_type]

        base_resources_all_versions = channel.sandbox.list_resources(resource_type="CONFIG_GENERATOR_REPO", limit=10000)
        base_resources_all_versions = [res for res in base_resources_all_versions if res.skynet_id and \
                'major_version' in res.attributes and \
                'repo_type' in res.attributes and res.attributes['repo_type'] == repo_type]

        major_version = self.ctx[ParamMajorVersion.name]
        is_incremental = self.ctx[ParamIsIncremental.name]
        base_resource = None

        if major_version == -1:
            if base_resources_all_versions:
                major_version = max([x.attributes["major_version"] for x in base_resources_all_versions])
                self.ctx[ParamMajorVersion.name] = major_version
            else:
                raise Exception, "Cannot determine version, please specify version!"
        base_resources = [x for x in base_resources_all_versions if x.attributes["major_version"] == major_version]

        if is_incremental:
            if base_resources:
                base_resource = max(base_resources, key=lambda x: x.task_id)
            else:
                base_resource = None
        else:
            base_resource = None
        is_incremental = base_resource is not None
        self.ctx[ParamIsIncremental.name] = is_incremental

        self.__start_build_resource()

        if base_resource:
            try:
                self.__download_base_resource(base_resource)
                self.__update_resource()
            except Exception:
                logging.info("Failed to download previous base %s, try with clone..." % base_resource.skynet_id)
                self.__clone_repo(repo_opts)
        else:
            self.__clone_repo(repo_opts)

        if base_resource:
            if "incremental_postactions" in repo_opts:
                self.__run_postactions(repo_opts["incremental_postactions"])
        else:
            if "full_postactions" in repo_opts:
                self.__run_postactions(repo_opts["full_postactions"])

        self.__finish_build_resource()

    def __start_build_resource(self):
        pass

    def __download_base_resource(self, base_resource):
        skynet_id = base_resource.skynet_id
        self.run_process(['sky', 'get', '-wu', skynet_id])
        resource_tar_path = self.abs_path(RESULT_TAR_FILE)
        if not os.path.exists(resource_tar_path):
            raise Exception("Resource \"%s\" has file doesn't have expected file" % skynet_id)
        result_dir = self.abs_path(RESULT_DIR)
        os.mkdir(result_dir)
        self.run_process(['tar', '-xzf', resource_tar_path, '-C', result_dir])

    def __clone_repo(self, repo_opts):
        result_path = self.abs_path(RESULT_DIR)
        os.mkdir(result_path)
        self.run_process(['git', 'clone', repo_opts['addr'], result_path])

    def __update_resource(self):
        self.run_process(['git', 'pull'], work_dir=self.abs_path(RESULT_DIR))

    def __run_postactions(self, postactions):
        result_path = self.abs_path(RESULT_DIR)
        for action in postactions:
            self.run_process(action, work_dir=result_path)

    def __finish_build_resource(self):
        self.run_process(["tar", "-czf", self.abs_path(RESULT_TAR_FILE), '-C', self.abs_path(RESULT_DIR), '.'])
        self.ctx['repo_resource_id'] = self._create_resource(
                self.descr, self.abs_path(RESULT_TAR_FILE),
                resource_types.CONFIG_GENERATOR_REPO, arch='any',
                attrs=dict(
                    repo_type=self.ctx[ParamRepoType.name],
                    major_version=self.ctx[ParamMajorVersion.name],
                    is_incremental=self.ctx[ParamIsIncremental.name],
                )
            ).id

    def run_process(self, *args, **kwargs):
        sdk_process.run_process(log_prefix="run_process_log", *args, **kwargs)


__Task__ = BuildConfigGeneratorRepo
