# -*- coding: utf-8 -*-

import os

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.gencfg.task import IGencfgBuildTask
from sandbox.projects.common.gencfg import utils as config_generator_utils

GENERATOR_DIR_NAME = 'generator'

SVN_COMMIT_KEY = 'svn_commit'


class GeneratorGitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Tag name (from /tags in repo)'
    default_value = 'trunk'


class BuildConfigGeneratorService(IGencfgBuildTask):
    type = 'BUILD_CONFIG_GENERATOR_SERVICE'

    input_parameters = [GeneratorGitTagParameter]

    def get_generator_path(self):
        return self.abs_path(GENERATOR_DIR_NAME)

    def get_db_path(self):
        return os.path.join(self.get_generator_path(), 'db')

    def get_balancer_path(self):
        return os.path.join(self.get_generator_path(), 'custom_generators', 'balancer_gencfg')

    def get_generator_archive_path(self):
        return self.abs_path('generator.tar.gz')

    def arcadia_info(self):
        return self.ctx.get(SVN_COMMIT_KEY, None), 'gencfg/%s' % (self.ctx[GeneratorGitTagParameter.name]), None

    def on_enqueue(self):
        self.ctx['gencfg_service_id'] = self._create_resource(self.descr, self.get_generator_archive_path(),
                                                               resource_types.CONFIG_GENERATOR_SERVICE, arch='linux').id

    def on_execute(self):
        config_generator_utils.clone_gencfg_all(self.get_generator_path(), 'full', tag=self.ctx[GeneratorGitTagParameter.name])

        # create resource with db from master, not from tag
        run_process(['svn', 'switch', '^/trunk/data/gencfg_db', '--non-interactive'], work_dir=os.path.join(self.get_generator_path(), 'db'), log_prefix='checkout_master_repo')
        run_process(['tar', '-czf', self.get_generator_archive_path(), '.'],
                    work_dir=os.path.join(self.abs_path(), GENERATOR_DIR_NAME), log_prefix='build_archive')
        if self.ctx[GeneratorGitTagParameter.name] != "trunk":
            run_process(['svn', 'switch', "^/tags/gencfg/%s/db" % self.ctx[GeneratorGitTagParameter.name], "--non-interactive"],
                        work_dir=os.path.join(self.get_generator_path(), 'db'), log_prefix='checkout_master_repo')

        self.ctx[SVN_COMMIT_KEY] = self.get_last_commit(self.get_generator_path())

        run_process(["bash", "./install.sh", "tests"], work_dir=self.get_generator_path(), log_prefix="install")

        # generate diff
        generate_diff_cmd = self.generate_diff_create_cmd(self.get_generator_path(), ["commit"], "^service-stable-(\d+)-r(\d+)$", commit_filter="lambda x: 'backend' in x.labels")
        run_process(generate_diff_cmd, work_dir=self.get_generator_path(), log_prefix="diff_builder")
        self.generate_diff_process_result()

    def on_release(self, additional_parameters):
        release_status = additional_parameters.get('release_status')
        release_subject = additional_parameters.get('release_subject')

        self.report_mail_on_release(release_status, release_subject)
        self.report_startrek_on_release(self.ctx[GeneratorGitTagParameter.name])

    @property
    def release_template(self):
        return self.ReleaseTemplate(message=self.ctx['generated_changelog'], subject=self.ctx[GeneratorGitTagParameter.name])


__Task__ = BuildConfigGeneratorService
