# coding: utf-8

import os
import logging
import re
import shutil
import time
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class BuildTestConveyorDashboard(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Conveyor dashboard viewer.
        Checkouts conveyor dashboard from SVN and
        creates virtualenv with necessary packets.
    """
    type = 'BUILD_CONVEYOR_DASHBOARD_FRONT_PROXY'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [ArcadiaUrlParameter,
                        # GeodataUrlParameter,
                        # AdminkaUtilsGeoParameter
                        ]

    URL_CONVEYOR_DASHBOARD_FRONT_SVN = 'quality/relev_tools/conveyor_dashboard/distributed/front_proxy'

    COMMONS_PREFIX = 'quality/relev_tools/conveyor_dashboard/commons'

    URLS_ARCADIA_PACKAGES = [
        ("{0}/get_file.py".format(COMMONS_PREFIX), "get_file.py"),
        ("{0}/fmldb_handmade_orm.py".format(COMMONS_PREFIX), "fmldb_handmade_orm.py")
    ]

    PATH_PACKET = 'conveyor_dashboard_front_proxy'
    PATH_TGZ = 'conveyor_dashboard_front_proxy.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Conveyor dashboard {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * export conveyor dashboard front project
        * create virtualenv
        * install dependencies there
        * create tgz archive
        * create shard
        """

        import virtualenv

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "conveyor_dashboard_front_proxy")
        path_virtualenv = os.path.join(path_packet, "env")

        os.mkdir(path_packet)
        self._export_arcadia(self.URL_CONVEYOR_DASHBOARD_FRONT_SVN, path_checkout)

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)

        # self._prepare_geobase(path_checkout)
        self._install_libs_from_arcadia(path_virtualenv)
        self._install_dependencies(path_virtualenv, path_checkout)

        virtualenv.make_environment_relocatable(path_virtualenv)

        # with open(os.path.join(path_checkout, "adminka/default_settings.py"), "a") as f:
        #    f.write("\nADMINKA_REVISION = '%d'\n\n" % (int(revision)))

        self._make_resource(path_packet)
        self._create_shard()

    def _install_libs_from_arcadia(self, path_virtualenv):
        logging.info('export python libs from Arcadia')
        path_packages = os.path.join(path_virtualenv, 'lib', 'python2.7', 'site-packages')

        for pack in self.URLS_ARCADIA_PACKAGES:
            self._export_arcadia(pack[0],
                                 os.path.join(path_packages, pack[1]))

    def _install_dependencies(self, path_virtualenv, path_checkout):
        logging.info('install python libs')
        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')

        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', os.path.join(path_checkout, 'pip-requirements.txt')],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

    def _make_resource(self, path_packet):
        logging.info('create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            tar.dereference = True
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='conveyor dashboard front proxy tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.CONVEYOR_DASHBOARD_FRONT,
            arch='linux'
        )

    def _create_shard(self):
        shard_name = 'conveyor_dashboard_front_proxy-r{0}-{1}'.format(self.ctx['revision'], time.strftime("%Y%m%d_%H%M%S"))

        logging.info('create shard {0}'.format(shard_name))
        self.ctx['shard_name'] = shard_name

        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)

        shutil.copy(self.path(self.PATH_TGZ), shard_path)
        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_path, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.PATH_TGZ)
            )

        logging.info('create shard resource')
        self.create_resource(
            description='conveyor dashboard front proxy shard r{0}'.format(self.ctx['revision']),
            resource_path=shard_path,
            resource_type=resource_types.CONVEYOR_DASHBOARD_FRONT_SHARD,
            arch='linux'
        )

        # do initialize shard
        logging.info('initialize shard')
        run_process(
            ['perl', utils.get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)

    # def _download_file(self, url, path):
    #    logging.info("DOWNLOAD {} to {}".format(url, path))
    #    if url[:8] == "rsync://":
    #        run_process(
    #            ['rsync', url, path],
    #            log_prefix="rsync-download",
    #            work_dir=self.path()
    #        )
    #    elif url[:10] == "svn+ssh://":
    #        Svn.export(url, path)
    #    else:
    #        urllib.urlretrieve(url, path)


__Task__ = BuildTestConveyorDashboard
