# coding: utf-8

import os
import logging
import re
import shutil
import time
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"

# class GeodataUrlParameter(SandboxStringParameter):
#    name = 'geodata_url'
#    description = 'URL to geodata3.bin'
#    default_value = "rsync://veles.yandex.ru/Berkanavt/geodata/geodata3.bin"

# class AdminkaUtilsGeoParameter(LastReleasedResource):
#    name = 'utils_geo'
#    description = 'Utility extract_geo_tree'
#    resource_type = resource_types.EXPERIMENTS_ADMINKA_UTILS_GEO


class BuildTestConveyorDashboard(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Conveyor dashboard viewer.
        Checkouts conveyor dashboard from SVN and
        creates virtualenv with necessary packets.
    """
    type = 'BUILD_CONVEYOR_DASHBOARD_MK2_TEST'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [ArcadiaUrlParameter,
                        # GeodataUrlParameter,
                        # AdminkaUtilsGeoParameter
                        ]

    URL_ADMINKA_SVN = 'quality/relev_tools/conveyor_dashboard/dashboard_mk3_test'

    COMMONS_PREFIX = 'quality/relev_tools/conveyor_dashboard/commons'

    URLS_ARCADIA_PACKAGES = [
        ("{0}/ALGORC_125_report_core.py".format(COMMONS_PREFIX), "ALGORC_125_report_core.py"),
        ("{0}/conveyor_status.py".format(COMMONS_PREFIX), "conveyor_status.py"),
        ("{0}/FML_2575_helper.py".format(COMMONS_PREFIX), "FML_2575_helper.py"),
        ("{0}/fml_progress_stages.py".format(COMMONS_PREFIX), "fml_progress_stages.py"),
        ("{0}/fml_progress_wrapper.py".format(COMMONS_PREFIX), "fml_progress_wrapper.py"),
        ("{0}/fml_xml_parser.py".format(COMMONS_PREFIX), "fml_xml_parser.py"),
        ("{0}/get_file.py".format(COMMONS_PREFIX), "get_file.py"),
        ("{0}/get_file_strategy.py".format(COMMONS_PREFIX), "get_file_strategy.py"),
        ("{0}/key_value_cache.py".format(COMMONS_PREFIX), "key_value_cache.py"),
        ("{0}/nirvana_api.py".format(COMMONS_PREFIX), "nirvana_api.py"),
        ("{0}/sqlite3_helpers.py".format(COMMONS_PREFIX), "sqlite3_helpers.py"),
        ("{0}/startrack_api.py".format(COMMONS_PREFIX), "startrack_api.py"),
        ("{0}/fml_helper.py".format(COMMONS_PREFIX), "fml_helper.py"),
        ("{0}/yed_graphml_bicycle.py".format(COMMONS_PREFIX), "yed_graphml_bicycle.py"),
        ("{0}/fmldb_handmade_orm.py".format(COMMONS_PREFIX), "fmldb_handmade_orm.py")
    ]

    PATH_PACKET = 'conveyor_dashboard'
    PATH_TGZ = 'conveyor_dashboard.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Conveyor dashboard {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * export adminka project
        * create virtualenv
        * install dependencies there
        * create tgz archive
        * create shard
        """

        import virtualenv

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "conveyor_dashboard")
        path_virtualenv = os.path.join(path_packet, "env")

        os.mkdir(path_packet)
        self._export_arcadia(self.URL_ADMINKA_SVN, path_checkout)

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)

        # self._prepare_geobase(path_checkout)
        self._install_libs_from_arcadia(path_virtualenv)
        self._install_dependencies(path_virtualenv, path_checkout)

        virtualenv.make_environment_relocatable(path_virtualenv)

        # with open(os.path.join(path_checkout, "adminka/default_settings.py"), "a") as f:
        #    f.write("\nADMINKA_REVISION = '%d'\n\n" % (int(revision)))

        self._make_resource(path_packet)
        self._create_shard()

    def _install_libs_from_arcadia(self, path_virtualenv):
        logging.info('export python libs from Arcadia')
        path_packages = os.path.join(path_virtualenv, 'lib', 'python2.7', 'site-packages')

        for pack in self.URLS_ARCADIA_PACKAGES:
            self._export_arcadia(pack[0],
                                 os.path.join(path_packages, pack[1]))

    def _install_dependencies(self, path_virtualenv, path_checkout):
        logging.info('install python libs')
        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')

        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', os.path.join(path_checkout, 'pip-requirements.txt')],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

    # def _prepare_geobase(self, path_checkout):
    #    path_geodata = self.path("geodata3.bin")
    #    path_geodata_tsv = os.path.join(path_checkout, "adminka", "geodata3.tsv")
    #
    #    url_geodata = self.ctx[GeodataUrlParameter.name]
    #
    #    logging.info("Extract geodata {}".format(url_geodata))
    #    self._download_file(url_geodata, path_geodata)
    #
    #    path_extracter = channel.task.sync_resource(self.ctx[AdminkaUtilsGeoParameter.name])
    #
    #    run_process(
    #        [path_extracter,
    #         "-g", path_geodata,
    #         "-o", path_geodata_tsv],
    #        log_prefix="extract-geobase",
    #        work_dir=self.path()
    #    )
    #
    #    return path_geodata_tsv

    def _make_resource(self, path_packet):
        logging.info('create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            tar.dereference = True
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='conveyor dashboard mk2 tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.CONVEYOR_DASHBOARD,
            arch='linux'
        )

    def _create_shard(self):
        shard_name = 'conveyor_dashboard_mk2-r{0}-{1}'.format(self.ctx['revision'], time.strftime("%Y%m%d_%H%M%S"))

        logging.info('create shard {0}'.format(shard_name))
        self.ctx['shard_name'] = shard_name

        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)

        shutil.copy(self.path(self.PATH_TGZ), shard_path)
        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_path, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.PATH_TGZ)
            )

        logging.info('create shard resource')
        self.create_resource(
            description='conveyor dashboard shard r{0}'.format(self.ctx['revision']),
            resource_path=shard_path,
            resource_type=resource_types.CONVEYOR_DASHBOARD_SHARD,
            arch='linux'
        )

        # do initialize shard
        logging.info('initialize shard')
        run_process(
            ['perl', utils.get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)

    # def _download_file(self, url, path):
    #    logging.info("DOWNLOAD {} to {}".format(url, path))
    #    if url[:8] == "rsync://":
    #        run_process(
    #            ['rsync', url, path],
    #            log_prefix="rsync-download",
    #            work_dir=self.path()
    #        )
    #    elif url[:10] == "svn+ssh://":
    #        Svn.export(url, path)
    #    else:
    #        urllib.urlretrieve(url, path)


__Task__ = BuildTestConveyorDashboard
