# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common.build.parameters import ArcadiaUrl

import logging
import os
import tarfile


class PCodePath(ArcadiaUrl):
    name, description = 'code_path', 'Code path'
    default_value = Arcadia.trunk_url('junk/anikella/cores')


class BuildCoresView(SandboxTask):
    type = 'BUILD_CORES_VIEW'
    input_parameters = [PCodePath]
    PROJECT_NAME = 'unknown_project'
    ENV_RTYPE = resource_types.UNKNOWN_RESOURCE
    TAR_RTYPE = resource_types.UNKNOWN_RESOURCE

    def on_execute(self):
        arc_code_path = self.ctx.get(PCodePath.name)
        parsed_url = Arcadia.parse_url(arc_code_path)
        self.PROJECT_NAME = parsed_url.path.strip('/').split('/')[-1]
        if self.PROJECT_NAME == 'cores':
            self.ENV_RTYPE = resource_types.CORES_VIRTUAL_ENV
            self.TAR_RTYPE = resource_types.CORES_CODE_TAR
        elif self.PROJECT_NAME == 'plots':
            self.ENV_RTYPE = resource_types.SAAS_PLOTS_VIRTUAL_ENV
            self.TAR_RTYPE = resource_types.SAAS_PLOTS_CODE_TAR
        elif self.PROJECT_NAME == 'saas_deploy':
            self.ENV_RTYPE = resource_types.SAAS_MONITOR_VIRTUAL_ENV
            self.TAR_RTYPE = resource_types.SAAS_MONITOR_CODE_TAR

        code_path = self.path(self.PROJECT_NAME)
        if arc_code_path == Arcadia.trunk_url():
            raise SandboxTaskFailureError('source code path must be specified')
        Arcadia.checkout(arc_code_path, code_path)

        logging.info('start building')
        req_path = os.path.join(code_path, 'pip-requirements.txt')
        if not os.path.exists(req_path):
            raise SandboxTaskFailureError('requirements not found: %s, dir_content: %s'
                                          % (req_path, os.listdir(code_path)))
        env_init_dir = self.path('env')
        venv = VirtualEnvironment(env_init_dir)
        if 'sys' in self.descr:
            venv.VIRTUALENV_EXE += ' --python /usr/bin/python2'
        with venv as venv:
            logging.info(venv.get_module_version('pip'))
            logging.info(venv.get_module_version('setuptools'))
            venv.pip('pip==9.0.3')
            venv.pip('setuptools==39.0.1')
            venv.pip('-r ' + req_path)
            venv.pack_bundle(self.path('env.tar.gz'))

        self._create_resource('Virtual env for ' + self.PROJECT_NAME + ', ' + self.descr,
                              self.path('env.tar.gz'), self.ENV_RTYPE)

        code_tar = self.path(self.PROJECT_NAME + '.tar.gz')
        with tarfile.open(code_tar, 'w:gz') as tf:
            tf.add(code_path, arcname=self.PROJECT_NAME)
        self._create_resource('source code for ' + self.PROJECT_NAME + ', ' + self.descr,
                              code_tar, self.TAR_RTYPE)


__Task__ = BuildCoresView
