# -*- coding: utf-8 -*-

import os
import time
import logging
import json
import tempfile
from datetime import datetime
from string import Template

from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import make_folder, copy_path, copytree3
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask


class PSvnUrl(SandboxStringParameter):
    default_value = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/'
    name, description = 'svn_url', 'Svn url for source'


class PTarget(SandboxStringParameter):
    name, description, default_value = 'target', 'Target', 'saas/rtyserver_test'


class PBinary(SandboxStringParameter):
    name, description, default_value = 'binary', 'Binary', 'rtyserver_test'


class PCmakeParameters(SandboxStringParameter):
    name, description = 'cmake_parameters', 'Cmake additional parameters'
    default_value = '-DUSE_PYTHON=NO -DGCC_LIBDIR=/usr/lib64'


class PPlatformParameter(SandboxStringParameter):
    name, description = 'build_platform', 'Gcc or clang'
    default_value = ''


class PTestScript(SandboxStringParameter):
    multiline = True
    name, description = 'tests_script', 'Tests script'


class PTargetFolder(SandboxStringParameter):
    name, description, default_value = 'target_folder', 'Coverage for arcadia subpath', 'saas'


class PRtyserverId(LastReleasedResource):
    required, resource_type = False, resource_types.RTYSERVER
    name, description = 'rtyserver_resource_id', 'Rtyserver binary'
    default_value = 0


class PSearchproxyId(LastReleasedResource):
    required, resource_type = False, resource_types.RTYSERVER_SEARCHPROXY
    name, description = 'searchproxy_resource_id', 'Searchproxy binary'
    default_value = 0


class PIndexerproxyId(LastReleasedResource):
    required, resource_type = False, resource_types.RTYSERVER_INDEXER_PROXY
    name, description = 'indexerproxy_resource_id', 'Indexerproxy binary'
    default_value = 0


class BuildCoverage(ArcadiaTask):
    type = 'BUILD_COVERAGE'

    input_parameters = [build_params.ArcadiaUrl, build_params.BuildSystem,
                        PTarget, PBinary,
                        PCmakeParameters,
                        PTestScript,
                        PTargetFolder,
                        build_params.ArcadiaPatch,
                        PPlatformParameter,
                        PRtyserverId, PSearchproxyId, PIndexerproxyId
                        ]

    environment = (SvnEnvironment(), )

    required_ram = 80000

    def initCtx(self):
        self.ctx['report_link'] = ''

    def getData(self):
        test_data_url = 'arcadia:/arc/trunk/arcadia_tests_data/rtyserver/test_data'
        test_data_path = ArcadiaTestData.get_arcadia_test_data(self, test_data_url)
        dict_url = 'arcadia:/arc/trunk/arcadia_tests_data/recognize'
        dict_path = ArcadiaTestData.get_arcadia_test_data(self, dict_url)
        dict_path = os.path.join(dict_path, 'dict.dict')
        return test_data_path, dict_path

    def prepareConfigs(self, arcadia_src_dir):
        conf_res_path = os.path.join(arcadia_src_dir, 'saas/rtyserver_test/func/configs')
        conf_path = os.path.join(self.log_path(), 'configs_u')
        copy_path(conf_res_path, conf_path)
        try:
            dm_conf = os.path.join(conf_path, 'deploy_manager')
            cl_conf = os.path.join(conf_path, 'cluster')
            logging.debug('dmconf, %s files found' % len(os.listdir(dm_conf)))
            logging.debug('clust_conf, %s files found' % len(os.listdir(cl_conf)))
            for f in [os.path.join(dm_conf, p) for p in os.listdir(dm_conf)] +\
                    [os.path.join(cl_conf, p) for p in os.listdir(cl_conf)]:
                if not os.path.isfile(f):
                    continue
                try:
                    logging.debug('start rewrite %s' % f)
                    with open(f, 'r') as cf:
                        cf_text = cf.read()
                    cf_j = json.loads(cf_text)
                    for comp in cf_j:
                        if 'external' in comp:
                            comp['external'] = False
                    with open(f, 'w') as cf:
                        cf.write(json.dumps(cf_j))
                    logging.debug('finish rewrite %s' % f)
                except Exception as e:
                    logging.error('while processing dm_conf %s, error %s' % (f, e))
        except Exception as e:
            logging.error(e)
        return conf_path

    def prepareBinaries(self, bin_dir):
        for b_name in ['rtyserver', 'searchproxy', 'indexerproxy']:
            b_id = self.ctx.get(b_name + '_resource_id')
            if not b_id:
                logging.warning('%s resource not set' % b_name)
                continue
            b_path = self.sync_resource(b_id)
            copy_path(b_path, os.path.join(bin_dir, b_name))

    def build_ymake(self, arcadia_src_dir, build_coverage_dir):
        flags = sdk.parse_flags(self.ctx.get('cmake_parameters', ''))
        if 'gcnohack' in self.descr:
            flags['USE_PYTHON'] = 'NO --add-result gcno'
        if 'trysdk' not in self.descr:
            build_dir = tempfile.mkdtemp()
            flags = self.ctx.get('cmake_parameters').strip().split()
            if 'gcc' in self.ctx.get('build_platform', ''):
                flags.append('--target-platform=' + self.ctx['build_platform'])
            if 'nogcno' not in self.descr:
                flags.append('--add-result=gcno')
            ymake_cmd = [os.path.join(arcadia_src_dir, 'ya'), 'make', '-xx', '-j', '32',
                         '--results-root', build_coverage_dir,
                         '--install', os.path.join(build_coverage_dir, 'bin'),
                         '--build-dir', build_dir,
                         '--build', 'coverage',
                         '-DWITHOUT_SYMLINKS=yes',
                         os.path.join(arcadia_src_dir, self.ctx['target'])] + flags
            run_process(ymake_cmd, wait=True, check=True, log_prefix='ymake')
            self.build_dir = build_dir
        else:
            sdk.do_build(
                    consts.YMAKE_BUILD_SYSTEM, arcadia_src_dir, [os.path.join(arcadia_src_dir, self.ctx['target'])],
                    clear_build=True,
                    build_type='coverage', def_flags=flags, results_dir=build_coverage_dir
                )
        return build_coverage_dir

    def on_execute(self):
        self.set_info('%s: execute started' % str(datetime.now()))

        arcadia_src_dir = self.path('arcadia')
        self.ctx['arcadia_revision'] = Arcadia.info(self.ctx['checkout_arcadia_from_url'])['entry_revision']
        svn_dir = self.get_arcadia_src_dir()
        make_folder(arcadia_src_dir)
        for p in os.listdir(svn_dir):
            if '.svn' not in p and 'junk' not in p:
                if os.path.isfile(os.path.join(svn_dir, p)):
                    copy_path(os.path.join(svn_dir, p), os.path.join(arcadia_src_dir, p))
                else:
                    copytree3(os.path.join(svn_dir, p), os.path.join(arcadia_src_dir, p), symlinks=True)
       # copy_path(svn_dir, arcadia_src_dir)
       # Arcadia.export(self.ctx['checkout_arcadia_from_url'], arcadia_src_dir, self.ctx['arcadia_revision'])
        self.set_info('%s: svn getting completed' % str(datetime.now()))
        conf_path = self.prepareConfigs(arcadia_src_dir)

        Arcadia.apply_patch(arcadia_src_dir, self.ctx.get(build_params.ArcadiaPatch.name), self.abs_path())

        build_coverage_dir = self.path('build_coverage')
        if self.ctx.get('build_system') == consts.YMAKE_BUILD_SYSTEM:
            build_coverage_dir = self.build_ymake(arcadia_src_dir, build_coverage_dir)

        # create gcov dir
        gcov_dir = self.path('gcov')
        make_folder(gcov_dir)

        cache_dir = os.path.join(self.abs_path(), 'cache')
        make_folder(cache_dir)

        # run tests
        cvars = {'$ARCADIA': arcadia_src_dir}
        cov_env = dict(os.environ)
        if self.ctx.get('build_system') == consts.YMAKE_BUILD_SYSTEM:
            cov_env['GCOV_PREFIX'] = build_coverage_dir + '/'
            build_dir_level = len(tempfile.mkdtemp().strip('/').split('/')) + 2  # .../build_root/a23f5ed57.../
            cov_env['GCOV_PREFIX_STRIP'] = str(build_dir_level)
            logging.info('GCOV_PREFIX_STRIP set as %s' % build_dir_level)
        test_data_path, dict_path = self.getData()
        os.chdir(os.path.join(build_coverage_dir, 'bin'))
        self.prepareBinaries(os.path.join(build_coverage_dir, 'bin'))

        tests_all = self.ctx['tests_script'].replace('./rtyserver', '\n./rtyserver')
        tests_all = tests_all.replace('\r', '')
        tests = [t for t in tests_all.split('\n') if len(t) > 2]
        logging.info('starting tests: %s tests found for executing' % len(tests))
        tcnt = 0
        timelate = self.timestamp_start + self.ctx['kill_timeout'] - 3600
        self.ctx['failed_tests'] = []
        for test in tests:
            test_r = Template(test).safe_substitute(cvars)
            test_r = test_r.replace('$CACHE_DIR', cache_dir)
            test_r = test_r.replace('$TEST_ROOT', os.path.join(arcadia_src_dir,
                                                               'saas/rtyserver/rtyserver_test/func'))
            test_r = test_r.replace('$DICT_PATH', dict_path)
            test_r = test_r.replace('$TEST_DATA_PATH', test_data_path)
            test_r = test_r.replace('$LOG_PATH', self.log_path())
            test_r = test_r.replace('$CONF_PATH', conf_path)
            if test_r:
                tcnt += 1
                if time.time() > timelate:
                    self.set_info(
                        "Attention: too late, stopping tests. "
                        "Only {0} tests from {1} done".format(tcnt - 1, len(tests)))
                    break
                try:
                    run_process(
                        test_r,
                        timeout=900, check=True,
                        log_prefix='run_test_%s_%s' % (str(tcnt), test_r.split()[2]),
                        outputs_to_one_file=False,
                        environment=cov_env)
                except Exception as e:
                    logging.info('test %s: got exception %s' % (tcnt, e))
                    self.ctx['failed_tests'].append(test_r.split()[2])

        self.set_info('%s: tests completed' % str(datetime.now()))

        logging.info('coverage dir content: %s' % os.listdir(build_coverage_dir))

        # 1 gcov
        os.chdir(gcov_dir)

        gcov_retrieve_cmd = [os.path.join(arcadia_src_dir, 'ya'), 'tool', 'gcov', '--print-path']
        if 'gcc' in self.ctx.get('build_platform', ''):
            gcov_retrieve_cmd.append('--toolchain=' + self.ctx['build_platform'])
        p = run_process(gcov_retrieve_cmd, outs_to_pipe=True)
        o, e = p.communicate()
        gcov_path = o.strip()

        # gcov_cmd = '%s -p -o %s %s' % (gcov_name, os.path.join('../build_coverage', self.ctx['target'], 'CMakeFiles',
        #                                  self.ctx['binary'] + '.dir'), self.ctx['gcov_sourcefile'])
        #run_process(gcov_cmd, timeout=10800, log_prefix='gcov')

        #self.set_info('%s: gcov completed' % str(datetime.now()))

        rm_contrib_cmd = 'rm -rf {0}/contrib'.format(build_coverage_dir)
        run_process(rm_contrib_cmd, timeout=3600, log_prefix='rm_remove_contrib')

        rm_webcomp_cmd = 'rm -rf {0}/saas/web'.format(build_coverage_dir)
        run_process(rm_webcomp_cmd, timeout=3600, log_prefix='rm_remove_web')

        rm_ragel_cmd = "sh -c 'find {0} -name \*.rl|xargs -L1 dirname | " \
                       "sort -r|uniq|xargs rm -rf'".format(build_coverage_dir)
        run_process(rm_ragel_cmd, timeout=3600, log_prefix='rm_ragel_simple', shell=True)

        rm_ragel_cmd = "sh -c 'find {0} -name \*.rl -or -name \*.rl6|xargs -L1 dirname | " \
            "sed s/arcadia/build_coverage/ |sort -r|uniq|xargs rm -rf'".format(arcadia_src_dir)
        run_process(rm_ragel_cmd, timeout=3600, log_prefix='rm_ragel_main', shell=True)

        rm_info_cmd = "sh -c 'find {0} -name \*.info|xargs -L1 dirname | " \
            "sed s/arcadia/build_coverage/ | sort -r | uniq | xargs rm -rf'".format(arcadia_src_dir)
        run_process(rm_info_cmd, timeout=3600, log_prefix='rm_info_main', shell=True)

        # 2 lcov
        info_file_name = self.ctx['binary'] + '.info'
        lcov_path = os.path.join(arcadia_src_dir, 'check/robotcheck/optlib/gcov/lcov')
        lcov_cmd = '%s -g %s -d %s --no-long-names' \
                   ' --rc lcov_function_coverage=1 --rc lcov_branch_coverage=1 --rc geninfo_gcov_all_blocks=0' \
                   ' -t %s -c -o %s' \
                   % (lcov_path, gcov_path, os.path.join(build_coverage_dir, self.ctx.get('target_folder', '')),
                      self.ctx['binary'], info_file_name)
        run_process(lcov_cmd, timeout=7200, log_prefix='lcov')

        self.set_info('%s: lcov completed' % str(datetime.now()))

        if self.ctx.get('build_system') == consts.YMAKE_BUILD_SYSTEM:
            with open(os.path.join(gcov_dir, info_file_name + '.fixed'), 'w') as out:
                run_process(['sed', 's,%s/build_root/[^/]*/,,g' % self.build_dir, info_file_name], stdout=out,
                            log_prefix='subst', outputs_to_one_file=False)
            info_file_name += '.fixed'

        # 3 genhtml
        gengtml_cmd = '%s --rc lcov_function_coverage=1 --rc lcov_branch_coverage=1 --ignore-errors=source -o report %s' % \
                      (os.path.join('../arcadia', 'check/robotcheck/optlib/gcov/genhtml'), info_file_name)
        p = run_process(gengtml_cmd, timeout=7200, log_prefix='genhtml')

        self.set_info('%s: genhtml completed' % str(datetime.now()))

        dirs_cov = dict()
        with open(p.stdout_path, 'r') as f:
            for line in f.readlines():
                if line.startswith('dir='):
                    parts = line.split()
                    parts = [pt for pt in parts if '=' in pt]
                    dir_data = dict([tuple(pt.split('=', 1)) for pt in parts])
                    if dir_data['dir'] in dirs_cov or dir_data['dir'].startswith('place'):
                        continue
                    try:
                        total = int(dir_data.get('lines_total', 1))
                        hit = int(dir_data.get('lines_hit', 0))
                        dirs_cov[dir_data['dir']] = \
                            {'perc': round(100 * float(hit)/total, 1),
                             'total': total,
                             'hit': hit}
                    except Exception as e:
                        logging.error('%s' % e)
        with open(self.log_path('cov_stat_no_rec.json'), 'w') as f:
            f.write(json.dumps(dirs_cov, indent=4))

        dirs_summ = {'_ALL_': {'total': 0, 'hit': 0}}
        for cdir, data in dirs_cov.items():
            dirs = cdir.split('/')
            for i in range(len(dirs)):
                idir = '/'.join(dirs[:i+1])
                if idir not in dirs_summ:
                    dirs_summ[idir] = {'total': 0, 'hit': 0}
                dirs_summ[idir]['total'] += data.get('total', 0)
                dirs_summ[idir]['hit'] += data.get('hit', 0)
            dirs_summ['_ALL_']['total'] += data.get('total', 0)
            dirs_summ['_ALL_']['hit'] += data.get('hit', 0)
        for cdir, data in dirs_summ.items():
            if data['total'] == 0:
                cov_perc = 0
            else:
                cov_perc = round(100 * float(data['hit'])/data['total'], 1)
            dirs_summ[cdir]['perc'] = cov_perc
        summ_stat_file = self.log_path('cov_stat_summ.json')
        with open(summ_stat_file, 'w') as f:
            f.write(json.dumps(dirs_summ, indent=4))
        copy_path(summ_stat_file, os.path.join(gcov_dir, 'cov_stat_summ.json'))

        self.ctx['cov_total_perc'] = dirs_summ['_ALL_']['perc']

        important_paths = ['rtyserver', 'searchproxy', 'indexerproxy', 'deploy_manager']
        if self.ctx.get('target_folder', '').endswith('rtyserver'):
            for path in important_paths:
                self.ctx['cov_' + path] = dirs_summ.get(path, {}).get('perc', -1)

        report_res = self.create_resource(description=self.descr,
                             resource_path=gcov_dir,
                             resource_type="RTYSERVER_COVERAGE_RES"
                             )

        self.ctx['report_link'] = os.path.join(gcov_dir, 'report/index.html')
        self.set_info('<a href="{}">Build Coverage report</a>'.format(report_res.proxy_url + 'report/index.html'))
        os.chdir(self.abs_path())


__Task__ = BuildCoverage
