# coding: utf-8
import os
import multiprocessing as mp

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.common.types.misc import DnsType
from sandbox.common.types.client import Tag


class NginxSourceVersion(SandboxStringParameter):
    name = 'nginx_source_version'
    description = 'Version of Nginx'
    default_value = '1.12.1'


class LibreSslSourceVersion(SandboxStringParameter):
    name = 'libressl_source_version'
    description = 'Version of LibreSSL'
    default_value = '2.5.5'


class PcreSourceVersion(SandboxStringParameter):
    name = 'pcre_source_version'
    description = 'Version of PCRE'
    default_value = '8.41'


class BuildCustomNginxBinary(sdk_task.SandboxTask):
    """
    Build Nginx statically from sources.
    """

    type = "BUILD_CUSTOM_NGINX"
    dns = DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE

    input_parameters = [
        NginxSourceVersion,
        LibreSslSourceVersion,
        PcreSourceVersion
    ]

    def on_execute(self):
        nginx_source_url = 'https://nginx.org/download/nginx-%s.tar.gz'
        nginx_source_ver = self.ctx['nginx_source_version']

        libressl_source_url = 'http://ftp.openbsd.org/pub/OpenBSD/LibreSSL/libressl-%s.tar.gz'
        libressl_source_ver = self.ctx['libressl_source_version']

        pcre_source_url = 'ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-%s.tar.gz'
        pcre_source_ver = self.ctx['pcre_source_version']

        current_dir = os.path.abspath('./')
        libressl_dir = os.path.join(current_dir, 'libressl-%s' % libressl_source_ver)
        pcre_dir = os.path.join(current_dir, 'pcre-%s' % pcre_source_ver)
        nginx_dir = os.path.join(current_dir, 'nginx-%s' % nginx_source_ver)
        temporary_dir = os.path.join(current_dir, 'tmp')
        staticlibpcre_dir = os.path.join(temporary_dir, 'staticlibpcre')

        sdk_process.run_process(
            ['wget', '-q', libressl_source_url % libressl_source_ver],
            log_prefix='get_libressl'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'libressl-%s.tar.gz' % libressl_source_ver],
            log_prefix='tar_libressl'
        )

        sdk_process.run_process(
            ['wget', '-q', pcre_source_url % pcre_source_ver],
            log_prefix='get_pcre'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'pcre-%s.tar.gz' % pcre_source_ver],
            log_prefix='tar_pcre'
        )

        sdk_process.run_process(
            ['wget', '-q', nginx_source_url % nginx_source_ver],
            log_prefix='get_nginx'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'nginx-%s.tar.gz' % nginx_source_ver],
            log_prefix='tar_nginx'
        )

        os.chdir(pcre_dir)

        sdk_process.run_process(
            [
                './configure',
                '--prefix=%s' % staticlibpcre_dir,
                '--disable-shared',
                '--enable-static',
                '--enable-utf8',
                '--enable-jit'
            ],
            work_dir=pcre_dir,
            log_prefix='configure_pcre'
        )

        sdk_process.run_process(
            ['make'],
            work_dir=pcre_dir,
            log_prefix='make_pcre'
        )

        sdk_process.run_process(
            ['make', 'install'],
            work_dir=pcre_dir,
            log_prefix='make_pcre'
        )

        os.chdir(nginx_dir)

        sdk_process.run_process(
            ["./configure",
            "--prefix=.",
            "--with-cc-opt=-O2 -I {}/include".format(staticlibpcre_dir),
            "--with-ld-opt=-Wl,--no-as-needed -lrt -L{}/lib".format(staticlibpcre_dir),
            "--with-debug",
            "--with-http_stub_status_module",
            "--with-http_ssl_module",
            "--with-ipv6",
            "--with-threads",
            "--with-file-aio",
            "--with-stream",
            "--with-openssl={}".format(libressl_dir),
            ], work_dir=nginx_dir, log_prefix="configure_nginx"
        )

        sdk_process.run_process(
            [
                "make", "-j", str(mp.cpu_count())
            ],
            log_prefix="make_nginx",
            work_dir=nginx_dir
        )

        self.create_resource(
            arch='linux',
            resource_path=os.path.join(nginx_dir, "objs", "nginx"),
            resource_type=resource_types.NGINX_BINARY,
            description='nginx binary {}'.format(nginx_source_ver),
            )


__Task__ = BuildCustomNginxBinary
