import logging
import os
import tarfile

from sandbox.projects.resource_types import PYTHON_BUNDLE
from sandbox.projects.BuildPythonBundle2 import BuildPythonBundle2


class BuildDTracker(BuildPythonBundle2):
    type = 'BUILD_DTRACKER'

    @property
    def release_template(self):
        logging.info('==> saved changelog:\n{}'.format(self.exec_context.get('changelog')))
        return self.ReleaseTemplate(
            subject='DTracker v.{} from {}'.format(self.ctx['version'], self.ctx['revision']),
            message=self.ctx.get('changelog')
        )

    def pre_install(self):
        dtracker_src = self.exec_context['target_src']

        # Get version.
        with open(os.path.join(dtracker_src, 'VERSION'), 'r') as f:
            version = f.read().strip().replace('\n', '')
            self.ctx['version'] = version

        # Sandbox task id.
        self.ctx['dtracker_sandbox_task_id'] = str(self.id)
        self.exec_context['staticfiles_path'] = os.path.join(dtracker_src, 'static')

    def do_create_resource(self):
        # Add staticfiles to main resource.
        target_env = self.exec_context['target_path']  # already built target.tar with virtualenv inside

        # Add version to main resource.
        version_path = self.path('VERSION')
        logging.info('==> writing version ({}) to {}'.format(
            self.ctx['version'],
            version_path
        ))
        with open(version_path, 'w') as f:
            f.write(self.ctx['version'])

        # Add sandbox task id to main resource.
        sandbox_task_id_path = self.path('sandbox_task_id.txt')
        logging.info('==> writing sandbox task id ({}) to {}'.format(
            self.ctx['dtracker_sandbox_task_id'],
            sandbox_task_id_path
        ))
        with open(sandbox_task_id_path, 'w') as f:
            f.write(self.ctx['dtracker_sandbox_task_id'])

        # Add commit info to main resource.
        revision_path = self.path('revision.txt')
        logging.info('==> writing revision ({}) to {}'.format(
            self.ctx['revision'],
            revision_path
        ))
        with open(revision_path, 'w') as f:
            f.write(self.ctx['revision'])

        compressed_dtracker_with_static = self.path('bundle.tar')
        with tarfile.TarFile(compressed_dtracker_with_static, 'w') as t:
            t.add(revision_path, 'revision.txt')
            t.add(sandbox_task_id_path, 'sandbox_task_id.txt')
            t.add(self.exec_context['staticfiles_path'], 'static')
            t.add(target_env, 'ENV')
            t.add(version_path, 'VERSION')

        # Create bundle resource.
        self.create_resource(
            description='compressed Dtracker virtualenv with static files',
            resource_path=compressed_dtracker_with_static,
            resource_type=PYTHON_BUNDLE,
            attributes={
                'branch': self.ctx['git_branch'],
                'commit': self.ctx['last_commit'],
                'version': self.ctx['version'],
                'build': self.ctx['build']
            }
        )


__Task__ = BuildDTracker
