# -*- coding: utf-8 -*-

import logging
import os
import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.parameters import (
    ResourceSelector, ListRepeater, SandboxStringParameter, Container
)

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class ContainerParameter(Container):
    required = True
    default_value = '140787721'
    description = "Container"


class PackagedResource(ResourceSelector):
    name = "packaged_resource_id"
    description = "Resource to package inside docker"
    resource_type = []
    required = True


class RegistryTags(ListRepeater, SandboxStringParameter):
    name = 'registry_tags'
    description = 'Tags to publish image with'
    default_value = [
        'registry.ape.yandex.net/project/test:latest',
        'registry.ape.yandex.net/project/test:r0',
    ]
    required = True


class BuildDockerImage(SandboxTask):
    """ Task for building Docker Image from your resource """
    type = "BUILD_DOCKER_IMAGE"
    privileged = True
    dns = ctm.DnsType.DNS64
    input_parameters = [ContainerParameter, PackagedResource, RegistryTags]
    client_tags = Tag.LINUX_PRECISE | Tag.LINUX_TRUSTY

    def on_prepare(self):
        self.res_path = self.sync_resource(self.ctx[PackagedResource.name])
        self.path_to_dockerfile = os.path.join(self.res_path, 'Dockerfile')
        assert os.path.exists(self.path_to_dockerfile), \
            'Dockerfile should be places inside your package root'

    def on_execute(self):
        task_dir = os.path.dirname(os.path.realpath(__file__))
        docker_setup_script = os.path.join(task_dir, 'docker_setup.sh')
        self.sync_resource(self.ctx[PackagedResource.name])
        run_process(['bash', docker_setup_script, self.abs_path('docker_root')],
                    shell=True, outputs_to_one_file=True, log_prefix='docker_install')
        run_process(['docker', 'build', '-t', 'basis', self.res_path], shell=True,
                    outputs_to_one_file=True, log_prefix='docker_build')

        for tag in list(self.ctx[RegistryTags.name]):
            run_process(['docker', 'tag', 'basis', tag], shell=True, outputs_to_one_file=True,
                        log_prefix='docker_tag')
            run_process(['docker', 'push', tag], shell=True, outputs_to_one_file=True,
                        log_prefix='docker_push')

    def on_finish(self):
        pass


__Task__ = BuildDockerImage


def create_task_BuildDockerImage(parent_task, resource_id, registry_tags, description,
                                 execution_space=2 * 1024,
                                 lxc_layer=ContainerParameter.default_value):
    """
        Create BuildDockerImage subtask.
        :param parent_task: родительская задача для той, которую создаст метод
        :param resource_ids: id ресурса, который надо обернуть в контейнер
        :param registry_tags: список тагов, в которые надо запушить собранный образ
        :param description: описание создаваемой задачи
        :param execution_space: максимально разрешенное количество диска для задачи (в МБ)
        :param lxc_layer: номер ресурса с базовым слоем
    """
    sub_ctx = {
        PackagedResource.name: resource_id,
        RegistryTags.name: registry_tags,
        ContainerParameter.name: lxc_layer,
    }
    logging.info("create {}({})".format(BuildDockerImage.type, str(sub_ctx)))
    return parent_task.create_subtask(
        task_type=BuildDockerImage.type,
        description=description,
        input_parameters=sub_ctx,
        arch='linux_ubuntu_14.04_trusty',
        execution_space=execution_space
    )
