# -*- coding: utf-8 -*-

"""
new in 6.04:
* support {branch} and {revision} placeholders in Deploy releases
"""

import os
import re
import textwrap
import requests
import logging
import shutil
import subprocess

from distutils.dir_util import copy_tree

import sandbox.projects.common.build.YaPackage as YaPackage
import sandbox.projects.common.build.parameters as build_params

from sandbox.common.types import resource, client
from sandbox.common import errors
from sandbox.projects import resource_types
from sandbox.projects.common.apihelpers import get_task_resource_id
from sandbox.sandboxsdk import process, task, parameters, svn
from sandbox.sandboxsdk.channel import channel
from sandbox.sdk2.yav import Secret
import sandbox.common.types.misc as ctm
import sandbox.projects.common.constants as consts

from sandbox.projects.common.ya_deploy import release_integration

REGISTRY = 'registry.yandex.net'
TOKEN_PATTERN = r'\$\((?P<storage>vault|yav):(?P<owner_ver>[^:]+):(?P<name>[^:]+)\)'


def lazyprop(fn):
    attr_name = '_lazy_' + fn.__name__

    @property
    def _lazyprop(self):
        if not hasattr(self, attr_name):
            setattr(self, attr_name, fn(self))
        return getattr(self, attr_name)

    return _lazyprop


def run_command(command):
    logging.info('RUNNING COMMAND: {}'.format(command))
    proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout_data, stderr_data = proc.communicate()
    if proc.returncode != 0:
        message = '{} failed, with status code {}\nSTDOUT: {}\nSTDERR: {}\n'
        message = message.format(command, proc.returncode, stdout_data, stderr_data)
        raise RuntimeError(message)
    else:
        message = '{} finished successfully \nSTDOUT: \n{}\nSTDERR: \n{}\n'
        message = message.format(command, stdout_data, stderr_data)
        return message


class BuildDockerImageV6(task.SandboxTask, release_integration.ReleaseToYaDeployTask):
    class ContainerParameter(parameters.Container):
        platform = "linux_ubuntu_14.04_trusty"
        description = "Ubuntu trusty stable container"
        required = True

    class DockerSetupScript(parameters.ResourceSelector):
        ui = None
        name = "docker_setup_script"
        description = "Script for docker setup"
        resource_type = [resource_types.RELEASABLE_DUMMY]
        attrs = {"config_type": "docker_setup"}
        state = resource.State.READY
        default_value = '2039543624'
        required = True

    class DockerfileUrl(parameters.SandboxUrlParameter):
        name = 'docker_file_url'
        description = "Dockerfile url"
        default_value = None
        required = False
        group = 'Image parameters'

    class PackagedResource(parameters.ResourceSelector):
        name = "packaged_resource_id"
        description = "Resource to package inside docker"
        resource_type = []
        required = False
        group = 'Image parameters'

    class ResourceType(parameters.SandboxRadioParameter):
        resource_types = [
            'directory',
            'tarball',
        ]
        choices = [(choice, choice) for choice in resource_types]
        description = 'Resource type for choosing preparation procedures'
        default_value = 'directory'
        name = 'docker_resource_type'
        group = 'Image parameters'

    class CacheFromImageTag(parameters.SandboxStringParameter):
        name = 'cache_from_image_tag'
        description = '--cache-from image tag (if present, will be used as docker build --cache-from=<this>)'
        group = 'Image parameters'
        required = False

    class DockerBuildArgs(parameters.ListRepeater, parameters.SandboxStringParameter):
        name = 'docker_build_args'
        description = textwrap.dedent(
        '''Docker --build-arg option

        May be used with Vault or Yav:
            - $(vault:owner:name)
            - $(yav:version:name)

        Example:
            YT_TOKEN=$(vault:yazevnul:yazevnul-yt_token)
            NIRVANA_TOKEN=$(yav:sec-01daxhrvak3kbcd6fxhydp3esp:nirvana-secret)
        ''')
        group = 'Image parameters'
        required = False

    class ArcadiaUrl(build_params.ArcadiaUrl):
        required = False
        name = 'docker_package_checkout_arcadia_from_url'
        group = 'Build From Arcadia'

    class ArcadiaPatch(build_params.ArcadiaPatch):
        required = False
        name = 'docker_package_arcadia_patch'
        group = 'Build From Arcadia'

    class DockerPackageJsonParameter(YaPackage.PackagesParameter):
        required = False
        name = 'docker_package_json'
        group = 'Build From Arcadia'

    class UseArcadiaApiFuse(build_params.UseArcadiaApiFuse):
        required = False
        group = 'Build From Arcadia'

    class RegistryTags(parameters.ListRepeater, parameters.SandboxStringParameter):
        name = 'registry_tags'
        description = 'Tags to publish image with (registry.yandex.net/<this tags>), {branch} and {revision} macros can be used when building from Arcadia'
        group = 'Registry Parameters'
        required = False

    class VaultItemOwner(parameters.SandboxStringParameter):
        name = 'vault_item_owner'
        description = 'Vault item owner'
        group = 'Registry Parameters'

    class VaultItemName(parameters.SandboxStringParameter):
        name = 'vault_item_name'
        description = (
            'Vault item with oauth token for '
            'registry.yandex.net (vault item name)'
        )
        group = 'Registry Parameters'

    class RegistryLogin(parameters.SandboxStringParameter):
        name = 'registry_login'
        description = 'Yandex login to use with docker login'
        group = 'Registry Parameters'
        required = False

    input_parameters = [
        ContainerParameter,
        DockerSetupScript,
        DockerfileUrl,
        PackagedResource,
        ResourceType,
        CacheFromImageTag,
        DockerBuildArgs,
        ArcadiaUrl,
        ArcadiaPatch,
        build_params.BuildType,
        DockerPackageJsonParameter,
        UseArcadiaApiFuse,
        RegistryTags,
        RegistryLogin,
        VaultItemName,
        VaultItemOwner,
        release_integration.ReleaseToYaDeployParameter,
        release_integration.YpTokenVaultParameter,
        release_integration.ReleaseTypeParameter,
    ]

    type = "BUILD_DOCKER_IMAGE_V6"
    privileged = True
    execution_space = 10000
    client_tags = client.Tag.IPV6 & client.Tag.LINUX_TRUSTY & ~client.Tag.MYT
    max_restarts = 0
    dns = ctm.DnsType.DNS64

    __package_root_name = 'package_new_root'
    __docker_root_name = 'docker_root'

    __package_resource_revision = None
    __package_resource_branch = None

    def on_prepare(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    assert not tag.startswith('registry.ape.yandex.net'), \
                        'registry.ape.yandex.net is deprecated'
                    assert self.oauth_token, 'Oauth token must not be None or False'

        self.__info_message(self.__docker_sock_p)

        assert os.path.exists(self.res_path), \
            'Dockerfile should be placed inside your package\'s root'

        assert os.path.exists(self.path_to_dockerfile_p), \
            'Dockerfile should be placed inside your package\'s root'

    def on_execute(self):
        self.__prepare_docker_engine()
        self.__docker_login()
        self.__build_docker_image()
        self.__docker_tag_and_push()
        self.__create_ya_deploy_release()

    def __get_token_var(self, var):
        """ Returns var={token} if TOKEN_PATTERN matches, otherwise returns var without changes """
        if '=' not in var:
            return var
        name, value = var.split('=', 1)
        m = re.match(TOKEN_PATTERN, value)
        if m is None:
            return var
        if m.group('storage') == 'vault':
            value = self.get_vault_data(m.group('owner_ver'), m.group('name'))
        else:  # yav
            value = Secret(m.group('owner_ver')).data()[m.group('name')]
        return "{}={}".format(name, value)

    def __prepare_docker_engine(self):
        process.run_process(list(self.__docker_setup_cmd),
                            shell=True, outputs_to_one_file=True, log_prefix='docker_install')

    def __docker_login(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    assert self.__is_logged_in, 'You must be logged in with docker login'
                    break

    def __build_docker_image(self):
        build_args = self.__docker_build_cmd
        docker_build_args = self.ctx.get(self.DockerBuildArgs.name)
        if docker_build_args:
            for barg in docker_build_args:
                build_args += ('--build-arg', self.__get_token_var(barg))
        cache_from_tag = self.ctx.get(self.CacheFromImageTag.name)
        if cache_from_tag:
            cache_from_tag = "{}/{}".format(REGISTRY, cache_from_tag)
            process.run_process(
                list(self.__docker_pull_cmd + (cache_from_tag,)),
                shell=True, outputs_to_one_file=True, log_prefix='docker_pull'
            )
            build_args += ('--cache-from', cache_from_tag)

        process.run_process(list(build_args),
                            shell=True, outputs_to_one_file=True, log_prefix='docker_build')

    def __docker_tag_and_push(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    tag = self.__apply_templates(tag)

                    if tag.startswith(REGISTRY):
                        registry_tag = tag
                    else:
                        registry_tag = "{}/{}".format(REGISTRY, tag)


                    self.__info_message(self.__docker_tag_cmd + (registry_tag,))
                    self.__info_message(self.__docker_push_cmd + (registry_tag,))

                    process.run_process(list(self.__docker_tag_cmd + (registry_tag,)),
                                        shell=True, outputs_to_one_file=True, log_prefix='docker_tag')

                    process.run_process(list(self.__docker_push_cmd + (registry_tag,)),
                                        shell=True, outputs_to_one_file=True, log_prefix='docker_push')

    def __get_dockerfile(self, path):
        docker_file_url = self.ctx[self.DockerfileUrl.name]

        if docker_file_url is None or len(str(docker_file_url)) == 0:
            return

        try:
            if docker_file_url.startswith("arcadia"):
                svn_url = svn.Arcadia.svn_url(docker_file_url)

                if not svn.Arcadia.check(svn_url):
                    raise errors.SandboxTaskFailureError('Sandbox branch {0} does not exist.'.format(svn_url))

                docker_file = svn.Arcadia.cat(svn_url)
                with open(path, 'wb') as fd:
                    fd.write(docker_file)
            else:
                r = requests.get(docker_file_url, verify=False, stream=True, timeout=60)
                with open(path, 'wb') as fd:
                    for chunk in r.iter_content(1024):
                        fd.write(chunk)

        except Exception as e:
            raise errors.TaskFailure(e)

    def __info_message(self, msg):
        logging.info(msg)
        self.set_info(msg)

    def __untar_docker_tarball(self, tarball_path):
        """
        :return: docker directory path
        """
        run_command(['mkdir', 'docker_dir'])
        run_command(['tar', '-xvf', tarball_path, '-C', 'docker_dir'])
        return os.path.join(os.getcwd(), 'docker_dir')

    def __save_package_resource_attrs(self, resource_id):
        self.__package_resource_revision = channel.sandbox.get_resource_attribute(resource_id, 'svn_revision')
        self.__package_resource_branch = channel.sandbox.get_resource_attribute(resource_id, 'branch').replace("/", ".")

    def __apply_templates(self, tag):
        result = tag

        if self.__package_resource_revision is not None:
            result = result.replace('{revision}', self.__package_resource_revision)
        if self.__package_resource_branch is not None:
            result = result.replace('{branch}', self.__package_resource_branch)

        return result

    def __create_ya_deploy_release(self):
        if not self.ctx.get(release_integration.ReleaseToYaDeployParameter.name):
            return
        release_type = self.ctx[release_integration.ReleaseTypeParameter.name]
        s = self.ctx[self.RegistryTags.name]
        # Will export to Ya.Deploy only first docker tag
        image = release_integration.make_docker_image_from_path(s[0])
        release_integration.create_docker_release_inner(
            task=self,
            images=[image],
            release_type=release_type,
            release_author=self.author,
            title='{}:{}'.format(image['name'], self.__apply_templates(image['tag'])),
            description=''
        )

    @lazyprop
    def res_path(self):
        if self.ctx.get(self.PackagedResource.name):
            resource_type = self.ctx.get(self.ResourceType.name)

            # RadioButton, no default/else clause needed
            if resource_type == 'directory':
                return self.sync_resource(self.ctx[self.PackagedResource.name])
            elif resource_type == 'tarball':
                tarball_path = self.sync_resource(self.ctx[self.PackagedResource.name])
                docker_dir_path = self.__untar_docker_tarball(tarball_path)
                return docker_dir_path

        elif self.ctx.get(self.DockerPackageJsonParameter.name):
            if not self.ctx.get('packaging_already_created'):
                checkout_mode = consts.CHECKOUT_MODE_MANUAL
                if self.ctx[self.ArcadiaUrl.name].startswith('arcadia:/arc/branches'):
                    checkout_mode = consts.CHECKOUT_MODE_AUTO

                subtask = self.create_subtask(
                    YaPackage.YaPackage.type,
                    'Packaging things for docker image for task {}'.format(self.id),
                    input_parameters={
                        # pass-through parameters from interface
                        build_params.ArcadiaUrl.name: self.ctx[self.ArcadiaUrl.name],
                        build_params.ArcadiaPatch.name: self.ctx[self.ArcadiaPatch.name],
                        build_params.UseArcadiaApiFuse.name: self.ctx.get(self.UseArcadiaApiFuse.name, False),
                        build_params.CheckoutModeParameter.name: checkout_mode,
                        YaPackage.PackagesParameter.name: self.ctx[self.DockerPackageJsonParameter.name],
                        build_params.BuildType.name: self.ctx[build_params.BuildType.name],
                        # consts
                        YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
                        YaPackage.CompressPackageArchiveParameter.name: True,
                        YaPackage.UseNewFormatParameter.name: True,
                        YaPackage.ResourceTypeParameter.name: resource_types.YA_PACKAGE.name,
                        YaPackage.PublishPackageParameter.name: False,
                    })
                self.ctx['packaging_already_created'] = True
                self.ctx['packaging_subtask_id'] = subtask.id
                self.wait_all_tasks_stop_executing([subtask])
            else:
                # YA_PACKAGE already done, second 'on_execute'
                resource_id = get_task_resource_id(self.ctx['packaging_subtask_id'], resource_types.YA_PACKAGE.name)

                self.__save_package_resource_attrs(resource_id)

                tarball_path = self.sync_resource(resource_id)
                docker_dir_path = self.__untar_docker_tarball(tarball_path)
                return docker_dir_path
        else:
            return self.__package_root_p

    @lazyprop
    def oauth_token(self):
        if self.token_owner and self.token_name:
            return self.get_vault_data(self.token_owner, self.token_name)
        elif self.token_name:
            return self.get_vault_data(self.token_name)
        else:
            return None

    @lazyprop
    def login(self):
        return self.ctx.get(self.RegistryLogin.name, self.author)

    @lazyprop
    def token_owner(self):
        return self.ctx.get(self.VaultItemOwner.name)

    @lazyprop
    def token_name(self):
        return self.ctx.get(self.VaultItemName.name)

    @lazyprop
    def __is_logged_in(self):
        env = os.environ.copy()
        env['REGISTRY_PASSWORD'] = self.oauth_token
        process.run_process(
            list(self.__docker_login_cmd),
            shell=True, outputs_to_one_file=True,
            log_prefix='docker_login', environment=env
        )
        return True

    @lazyprop
    def __docker_cmd(self):
        return tuple(('docker', '-H', self.__docker_sock_p))

    @lazyprop
    def __docker_tag_cmd(self):
        return self.__docker_cmd + ('tag', 'base',)

    @lazyprop
    def __docker_push_cmd(self):
        return self.__docker_cmd + ('push',)

    @lazyprop
    def __docker_setup_cmd(self):
        return tuple(('bash', self.__container_setup_script_p, self.__docker_root_p))

    @lazyprop
    def __docker_build_cmd(self):
        return self.__docker_cmd + ('build', '-t', 'base', self.res_path,)

    @lazyprop
    def __docker_login_cmd(self):
        return self.__docker_cmd + ('login', '-u', self.login, '-p', '$REGISTRY_PASSWORD', REGISTRY,)

    @lazyprop
    def __docker_pull_cmd(self):
        return self.__docker_cmd + ('pull',)

    @lazyprop
    def __package_root_p(self):
        self.__package_root = os.path.join(self.abs_path(), self.__package_root_name)
        self.__info_message('Package_root: ' + self.__package_root)
        try:
            os.makedirs(self.__package_root)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)
        return self.__package_root

    @lazyprop
    def __docker_root_p(self):
        self.__docker_root = os.path.join(self.abs_path(), self.__docker_root_name)
        try:
            os.makedirs(self.__docker_root)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)
        return self.__docker_root

    @lazyprop
    def __container_setup_script_p(self):
        setup_script = self.sync_resource(self.ctx[self.DockerSetupScript.name])
        new_setup_script = os.path.join(os.path.abspath(self.__package_root_p), 'setup_script')
        # Removing \r from docker_setup_script
        # Begin
        with open(new_setup_script, 'w') as wf:
            with open(setup_script) as f:
                for line in f.readlines():
                    wf.write(line.rstrip())
                    wf.write('\n')
        # End
        assert os.path.exists(new_setup_script), \
            'new_setup_script must exist'
        self.__container_setup_script = new_setup_script
        return self.__container_setup_script

    @lazyprop
    def __docker_sock_p(self):
        self.__docker_sock = "unix://{}".format(os.path.join(self.__docker_root_p, 'docker.sock'))
        return self.__docker_sock

    @lazyprop
    def path_to_dockerfile_p(self):
        path_to_dockerfile = None

        # if fields was manually cleared
        package_resource = self.ctx.get(self.PackagedResource.name)
        if package_resource is not None and len(str(package_resource)) == 0:
            package_resource = None

        dockerfile_url = self.ctx.get(self.DockerfileUrl.name)
        if dockerfile_url is not None and len(str(dockerfile_url)) == 0:
            dockerfile_url = None

        yamake_resource = self.ctx.get(self.DockerPackageJsonParameter.name)
        if yamake_resource is not None and len(str(yamake_resource)) == 0:
            yamake_resource = None

        if package_resource is not None or yamake_resource is not None:
            copy_tree(self.res_path, self.__package_root_p, preserve_symlinks=1)
            logging.info('Found resource dir {0}. Copy to {1}'.format(self.res_path, self.__package_root_p))

            docker_file = os.path.join(self.res_path, 'Dockerfile')
            if os.path.exists(docker_file):
                path_to_dockerfile = docker_file
                logging.info('Dockerfile path: {0}'.format(path_to_dockerfile))

        assert (dockerfile_url is not None and path_to_dockerfile is None) \
            or (dockerfile_url is None), \
            "Ambiguities definition of Dockerfile. Remove Dockerfile url or Dockerfile from package"

        if dockerfile_url is not None:
            setattr(self, '_lazy_res_path', self.__package_root_p)
            path_to_dockerfile = os.path.join(self.__package_root_p, 'Dockerfile')
            self.__get_dockerfile(path_to_dockerfile)

        assert path_to_dockerfile is not None, "No Dockerfile was found"

        return path_to_dockerfile

    @staticmethod
    def copytree(src, dst):
        try:
            shutil.copytree(src, dst)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)

    prepare_docker_engine = __prepare_docker_engine
    docker_login = __docker_login
    build_docker_image = __build_docker_image
    docker_tag_and_push = __docker_tag_and_push
    is_logged_in = __is_logged_in
    docker_cmd = __docker_cmd


__Task__ = BuildDockerImageV6
