# -*- coding: utf-8 -*-

"""
new in 6.01:
* ability to use tarball resource
"""

import os
import requests
import logging
import shutil
import subprocess

import sandbox.projects.common.build.parameters
import sandbox.projects.common.build.YaPackage as YaPackage
from sandbox.projects import resource_types
from sandbox.common.types import resource, client
from sandbox.common import errors
from sandbox.sandboxsdk import process, task, parameters
import sandbox.common.types.misc as ctm

from sandbox.projects.common.apihelpers import get_task_resource_id

REGISTRY = 'registry.yandex.net'


def lazyprop(fn):
    attr_name = '_lazy_' + fn.__name__

    @property
    def _lazyprop(self):
        if not hasattr(self, attr_name):
            setattr(self, attr_name, fn(self))
        return getattr(self, attr_name)
    return _lazyprop


def run_command(command):
    logging.info('RUNNING COMMAND: {}'.format(command))
    proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout_data, stderr_data = proc.communicate()
    if proc.returncode != 0:
        message = '{} failed, with status code {}\nSTDOUT: {}\nSTDERR: {}\n'
        message = message.format(command, proc.returncode, stdout_data, stderr_data)
        raise RuntimeError(message)
    else:
        message = '{} finished successfully \nSTDOUT: \n{}\nSTDERR: \n{}\n'
        message = message.format(command, stdout_data, stderr_data)
        return message


class BuildDockerImageV6_01(task.SandboxTask):

    class ContainerParameter(parameters.Container):
        platform = "linux_ubuntu_14.04_trusty"
        description = "Ubuntu trusty stable container"
        required = True

    class DockerSetupScript(parameters.ResourceSelector):
        ui = None
        name = "docker_setup_script"
        description = "Script for docker setup"
        resource_type = [resource_types.RELEASABLE_DUMMY]
        attrs = {"config_type": "docker_setup"}
        state = resource.State.READY
        default_value = '2039543624'
        required = True

    class DockerfileUrl(parameters.SandboxUrlParameter):
        name = 'docker_file_url'
        description = "Dockerfile url"
        default_value = None
        required = False
        group = 'Image parameters'

    class PackagedResource(parameters.ResourceSelector):
        name = "packaged_resource_id"
        description = "Resource to package inside docker"
        resource_type = []
        required = False
        group = 'Image parameters'

    class ResourceType(parameters.SandboxRadioParameter):
        resource_types = [
            'directory',
            'tarball',
        ]
        choices = [(choice, choice) for choice in resource_types]
        description = 'Resource type for choosing preparation procedures'
        default_value = 'directory'
        name = 'resource_type'
        group = 'Image parameters'

    class ArcadiaUrl(sandbox.projects.common.build.parameters.ArcadiaUrl):
        required = False
        name = 'docker_package_checkout_arcadia_from_url'
        group = 'Build From Arcadia'

    class ArcadiaPatch(sandbox.projects.common.build.parameters.ArcadiaPatch):
        required = False
        name = 'docker_package_arcadia_patch'
        group = 'Build From Arcadia'

    class DockerPackageJsonParameter(YaPackage.PackagesParameter):
        required = False
        name = 'docker_package_json'
        group = 'Build From Arcadia'

    class RegistryTags(parameters.ListRepeater, parameters.SandboxStringParameter):
        name = 'registry_tags'
        description = 'Tags to publish image with (registry.yandex.net/<this tags>)'
        group = 'Registry Parameters'
        required = False

    class VaultItemOwner(parameters.SandboxStringParameter):
        name = 'vault_item_owner'
        description = 'Vault item owner'
        group = 'Registry Parameters'

    class VaultItemName(parameters.SandboxStringParameter):
        name = 'vault_item_name'
        description = (
            'Vault item with oauth token for '
            'registry.yandex.net (vault item name)'
        )
        group = 'Registry Parameters'

    class RegistryLogin(parameters.SandboxStringParameter):
        name = 'registry_login'
        description = 'Yandex login to use with docker login'
        group = 'Registry Parameters'
        required = False

    input_parameters = [
        ContainerParameter,
        DockerSetupScript,
        DockerfileUrl,
        PackagedResource,
        ResourceType,
        ArcadiaUrl,
        ArcadiaPatch,
        DockerPackageJsonParameter,
        RegistryTags,
        RegistryLogin,
        VaultItemName,
        VaultItemOwner
    ]

    type = "BUILD_DOCKER_IMAGE_V6_01"
    privileged = True
    execution_space = 10000
    client_tags = client.Tag.IPV6 & client.Tag.LINUX_TRUSTY
    max_restarts = 0
    dns = ctm.DnsType.DNS64

    __package_root_name = 'package_new_root'
    __docker_root_name = 'docker_root'

    def on_prepare(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    assert not tag.startswith('registry.ape.yandex.net'), \
                        'registry.ape.yandex.net is deprecated'
                    assert self.oauth_token, 'Oauth token must not be None or False'

        self.__info_message(self.__docker_sock_p)

        assert os.path.exists(self.res_path), \
            'Dockerfile should be placed inside your package\'s root'

        assert os.path.exists(self.path_to_dockerfile_p), \
            'Dockerfile should be placed inside your package\'s root'

    def on_execute(self):
        self.__prepare_docker_engine()
        self.__docker_login()
        self.__build_docker_image()
        self.__docker_tag_and_push()

    def __prepare_docker_engine(self):
        process.run_process(list(self.__docker_setup_cmd),
                            shell=True, outputs_to_one_file=True, log_prefix='docker_install')

    def __docker_login(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    assert self.__is_logged_in, 'You must be logged in with docker login'
                    break

    def __build_docker_image(self):
        process.run_process(list(self.__docker_build_cmd),
                            shell=True, outputs_to_one_file=True, log_prefix='docker_build')

    def __docker_tag_and_push(self):
        if self.ctx[self.RegistryTags.name] is not None:
            for tag in self.ctx[self.RegistryTags.name]:
                if tag:
                    if tag.startswith(REGISTRY):
                        registry_tag = tag
                    else:
                        registry_tag = "{}/{}".format(REGISTRY, tag)

                    self.__info_message(self.__docker_tag_cmd + (registry_tag,))
                    self.__info_message(self.__docker_push_cmd + (registry_tag,))

                    process.run_process(list(self.__docker_tag_cmd + (registry_tag,)),
                                        shell=True, outputs_to_one_file=True, log_prefix='docker_tag')

                    process.run_process(list(self.__docker_push_cmd + (registry_tag,)),
                                        shell=True, outputs_to_one_file=True, log_prefix='docker_push')

    def __get_dockerfile(self, path):
        try:
            r = requests.get(self.ctx[self.DockerfileUrl.name],
                             verify=False, stream=True, timeout=60)
            with open(path, 'wb') as fd:
                for chunk in r.iter_content(1024):
                    fd.write(chunk)

        except Exception as e:
            raise errors.TaskFailure(e)

    def __info_message(self, msg):
        logging.info(msg)
        self.set_info(msg)

    def __untar_docker_tarball(self, tarball_path):
        """
        :return: docker directory path
        """
        run_command(['mkdir', 'docker_dir'])
        run_command(['tar', '-xvf', tarball_path, '-C', 'docker_dir'])
        return os.path.join(os.getcwd(), 'docker_dir')

    @lazyprop
    def res_path(self):
        if self.ctx.get(self.PackagedResource.name):
            resource_type = self.ctx.get(self.ResourceType.name)

            # RadioButton, no default/else clause needed
            if resource_type == 'directory':
                return self.sync_resource(self.ctx[self.PackagedResource.name])
            elif resource_type == 'tarball':
                tarball_path = self.sync_resource(self.ctx[self.PackagedResource.name])
                docker_dir_path = self.__untar_docker_tarball(tarball_path)
                return docker_dir_path

        elif self.ctx.get(self.DockerPackageJsonParameter.name):
            if not self.ctx.get('packaging_already_created'):
                subtask = self.create_subtask(
                    YaPackage.YaPackage.type,
                    'Packaging things for docker image for task {}'.format(self.id),
                    input_parameters={
                        # pass-through parameters from interface
                        sandbox.projects.common.build.parameters.ArcadiaUrl.name: self.ctx[self.ArcadiaUrl.name],
                        sandbox.projects.common.build.parameters.ArcadiaPatch.name: self.ctx[self.ArcadiaPatch.name],
                        YaPackage.PackagesParameter.name: self.ctx[self.DockerPackageJsonParameter.name],
                        # consts
                        YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
                        sandbox.projects.common.build.parameters.BuildType.name: YaPackage.RELEASE,
                        YaPackage.CompressPackageArchiveParameter.name: True,
                        YaPackage.UseNewFormatParameter.name: True,
                        YaPackage.ResourceTypeParameter.name: resource_types.YA_PACKAGE.name,
                        YaPackage.PublishPackageParameter.name: False,
                        YaPackage.consts.USE_AAPI_FUSE: True,
                    })
                self.ctx['packaging_already_created'] = True
                self.ctx['packaging_subtask_id'] = subtask.id
                self.wait_all_tasks_stop_executing([subtask])
            else:
                # YA_PACKAGE already done, second 'on_execute'
                resource_id = get_task_resource_id(self.ctx['packaging_subtask_id'], resource_types.YA_PACKAGE.name)
                tarball_path = self.sync_resource(resource_id)
                docker_dir_path = self.__untar_docker_tarball(tarball_path)
                return docker_dir_path
        else:
            return self.__package_root_p

    @lazyprop
    def oauth_token(self):
        if self.token_owner and self.token_name:
            return self.get_vault_data(self.token_owner, self.token_name)
        elif self.token_name:
            return self.get_vault_data(self.token_name)
        else:
            return None

    @lazyprop
    def login(self):
        return self.ctx.get(self.RegistryLogin.name, self.author)

    @lazyprop
    def token_owner(self):
        return self.ctx.get(self.VaultItemOwner.name)

    @lazyprop
    def token_name(self):
        return self.ctx.get(self.VaultItemName.name)

    @lazyprop
    def __is_logged_in(self):
        env = os.environ.copy()
        env['REGISTRY_PASSWORD'] = self.oauth_token
        process.run_process(
            list(self.__docker_login_cmd),
            shell=True, outputs_to_one_file=True,
            log_prefix='docker_login', environment=env
        )
        return True

    @lazyprop
    def __docker_cmd(self):
        return tuple(('docker', '-H', self.__docker_sock_p))

    @lazyprop
    def __docker_tag_cmd(self):
        return self.__docker_cmd + ('tag', 'base', )

    @lazyprop
    def __docker_push_cmd(self):
        return self.__docker_cmd + ('push', )

    @lazyprop
    def __docker_setup_cmd(self):
        return tuple(('bash', self.__container_setup_script_p, self.__docker_root_p))

    @lazyprop
    def __docker_build_cmd(self):
        return self.__docker_cmd + ('build', '-t', 'base', self.res_path, )

    @lazyprop
    def __docker_login_cmd(self):
        return self.__docker_cmd + ('login', '-u', self.login, '-p', '$REGISTRY_PASSWORD', REGISTRY, )

    @lazyprop
    def __package_root_p(self):
        self.__package_root = os.path.join(self.abs_path(), self.__package_root_name)
        self.__info_message('Package_root: ' + self.__package_root)
        try:
            os.makedirs(self.__package_root)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)
        return self.__package_root

    @lazyprop
    def __docker_root_p(self):
        self.__docker_root = os.path.join(self.abs_path(), self.__docker_root_name)
        try:
            os.makedirs(self.__docker_root)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)
        return self.__docker_root

    @lazyprop
    def __container_setup_script_p(self):
        setup_script = self.sync_resource(self.ctx[self.DockerSetupScript.name])
        new_setup_script = os.path.join(os.path.abspath(self.__package_root_p), 'setup_script')
        # Removing \r from docker_setup_script
        # Begin
        with open(new_setup_script, 'w') as wf:
            with open(setup_script) as f:
                for line in f.readlines():
                    wf.write(line.rstrip())
                    wf.write('\n')
        # End
        assert os.path.exists(new_setup_script), \
            'new_setup_script must exist'
        self.__container_setup_script = new_setup_script
        return self.__container_setup_script

    @lazyprop
    def __docker_sock_p(self):
        self.__docker_sock = "unix://{}".format(os.path.join(self.__docker_root_p, 'docker.sock'))
        return self.__docker_sock

    @lazyprop
    def path_to_dockerfile_p(self):
        if self.ctx.get(self.DockerfileUrl.name) and self.ctx.get(self.PackagedResource.name):
            self.copytree(self.res_path, self.__package_root_p)
            setattr(self, '_lazy_res_path', self.__package_root_p)
            path_to_dockerfile = os.path.join(self.res_path, 'Dockerfile')
            self.__get_dockerfile(path_to_dockerfile)
        elif self.ctx.get(self.DockerfileUrl.name):
            setattr(self, '_lazy_res_path', self.__package_root_p)
            path_to_dockerfile = os.path.join(self.__package_root_p, 'Dockerfile')
            self.__get_dockerfile(path_to_dockerfile)
        else:
            path_to_dockerfile = os.path.join(self.res_path, 'Dockerfile')

        return path_to_dockerfile

    @staticmethod
    def copytree(src, dst):
        try:
            shutil.copytree(src, dst)
        except OSError as e:
            if e.errno != 17:
                raise errors.TaskFailure(e)

    prepare_docker_engine = __prepare_docker_engine
    docker_login = __docker_login
    build_docker_image = __build_docker_image
    docker_tag_and_push = __docker_tag_and_push
    is_logged_in = __is_logged_in
    docker_cmd = __docker_cmd


__Task__ = BuildDockerImageV6_01
