# coding: utf-8
import os
import requests
import zipfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter, SandboxStringParameter, SandboxBoolParameter


class GoResParam(SandboxIntegerParameter):
    name = 'go_resource'
    description = 'Resource with prebuilded GO archive'
    required = True
    default_value = 177483046


class GoVersion(SandboxStringParameter):
    name = 'go_source_version'
    description = 'Version of prebuilded GO for download from https://storage.googleapis.com/golang/. Without .tar.gz.'
    required = True
    default_value = 'go1.7.3.linux-amd64'


class EtcdVersion(SandboxStringParameter):
    name = 'etcd_source_version'
    description = "Version of etcd to get without '.zip', see https://github.com/coreos/etcd/releases"
    default_value = 'release-3.0'
    required = True


class EtcdResParam(SandboxIntegerParameter):
    name = 'etcd_resource'
    description = 'Resource with etcd sources archive'
    required = True
    default_value = 108893922


class GoUseResource(SandboxBoolParameter):
    name = 'go_use_resource'
    description = 'Use Sandbox resource as GO archive'
    default_value = True
    sub_fields = {
        'true':
            [GoResParam.name],
        'false':
            [GoVersion.name],
    }


class EtcdUseResource(SandboxBoolParameter):
    name = 'etcd_use_resource'
    description = 'Use Sandbox resource as etcd sources'
    default_value = False
    required = True
    sub_fields = {
        'false':
            [EtcdVersion.name],
        'true':
            [EtcdResParam.name]
    }


class BuildEtcd(SandboxTask):
    type = 'BUILD_ETCD'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_TRUSTY

    input_parameters = [
        GoUseResource,
        GoResParam,
        GoVersion,
        EtcdUseResource,
        EtcdVersion,
        EtcdResParam,
    ]

    _etcd_url = 'https://github.com/coreos/etcd/archive/'
    _go_url = 'https://storage.googleapis.com/golang/'

    def _get_etcd_sources(self, url=_etcd_url):
        if self.ctx.get('etcd_use_resource'):
            return self.sync_resource(self.ctx.get('etcd_resource'))
        else:
            filename = self.ctx.get('etcd_source_version') + '.zip'
            try:
                req = requests.get(url + filename)
                with open(filename, 'wb') as f:
                    f.write(req.content)
                    return filename
            except requests.exceptions.HTTPError as e:
                raise Exception("Could not get etcd sources. Raised: {}".format(e))

    def _get_go_dist(self, url=_go_url):
        if self.ctx.get('go_use_resource'):
            return self.sync_resource(self.ctx.get('go_resource'))
        else:
            filename = self.ctx.get('go_source_version') + '.tar.gz'
            try:
                req = requests.get(url + filename)
                with open(filename, 'wb') as f:
                    f.write(req.content)
                    return filename
            except requests.exceptions.HTTPError as e:
                raise Exception("Could not get etcd sources. Raised: {}".format(e))

    def on_execute(self):
        # get GO sources
        go_dist = self._get_go_dist()
        run_process(['tar', 'xvf', go_dist],
                    work_dir=self.path())
        goroot_path = self.path('go')  # we unpacked here
        go_bins = os.path.join(goroot_path, 'bin')

        # get etcd sources
        etcd_src = self._get_etcd_sources()
        zf = zipfile.ZipFile(etcd_src, 'r')
        etcd_src_dir = zf.namelist()[0]
        zf.close()  # zipfile.ZipFile.extractall couldn't extract symlinks (!!!)
        run_process(['unzip', etcd_src],
                    work_dir=self.path())
        etcd_bin_dir = os.path.join(etcd_src_dir, 'bin')

        env = dict(os.environ)
        env.update({'GOROOT': goroot_path, 'PATH': go_bins + ':' + env['PATH']})

        run_process(['bash', '-x', 'test'],
                    log_prefix='build-and-test',
                    work_dir=self.path(etcd_src_dir),
                    environment=env)

        run_process(['tar', 'czvf', 'etcd.tar.gz', 'etcd', 'etcdctl'],
                    work_dir=self.path(etcd_bin_dir))

        version = run_process(['./etcd', '-version'],
                              work_dir=self.path(etcd_bin_dir),
                              outs_to_pipe=True).stdout.read()

        for line in version.splitlines():
            (k, v) = line.split(': ', 1)
            if 'Version' in k:
                if 'etcd' in k:
                    self.ctx['etcd_version'] = v
                elif 'Go' in k:
                    self.ctx['go_version'] = v

        resources = []
        resource = self.create_resource(description='',
                                        resource_path=self.path(os.path.join(etcd_bin_dir, 'etcd')),
                                        resource_type=resource_types.ETCD_BIN,
                                        attributes={'ttl': '2',
                                                    'version': self.ctx.get('etcd_version'),
                                                    'go': self.ctx.get('go_version')},
                                        arch='linux')
        resources.append(resource.id)
        resource = self.create_resource(description='',
                                        resource_path=self.path(os.path.join(etcd_bin_dir, 'etcdctl')),
                                        resource_type=resource_types.ETCDCTL_BIN,
                                        attributes={'ttl': '2',
                                                    'version': self.ctx.get('etcd_version'),
                                                    'go': self.ctx.get('go_version')},
                                        arch='linux')
        resources.append(resource.id)
        resource = self.create_resource(description='',
                                        resource_path=self.path(os.path.join(etcd_bin_dir, 'etcd.tar.gz')),
                                        resource_type=resource_types.ETCD_PKG,
                                        attributes={'ttl': '2',
                                                    'version': self.ctx.get('etcd_version'),
                                                    'go': self.ctx.get('go_version')},
                                        arch='linux')
        resources.append(resource.id)

        for res in resources:
            self.mark_resource_ready(res)


__Task__ = BuildEtcd
