# coding: utf-8

import os
import logging
import re
import shutil
import time
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, ResourceSelector, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import copy_path, remove_path
from sandbox.sandboxsdk.channel import channel

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class AbtResourceSelector(ResourceSelector):
    name = 'abt_resource_id'
    description = 'Abt packets'
    resource_type = 'YA_PACKAGE'
#    resource_name = 'yandex-search-ab-testing'
    group = 'programms parameter'


class SpuResourceSelector(ResourceSelector):
    name = 'spu_resource_id'
    description = 'Spu packets'
    resource_type = 'YA_PACKAGE'
#    resource_name = 'yandex-search-ab-testing-spu-test'
    group = 'programms parameter'


class OddResourceSelector(ResourceSelector):
    name = 'odd_resource_id'
    description = 'Odd packets'
    resource_type = 'YA_PACKAGE'
#    resource_name = 'yandex-search-ab-testing-odd'
    group = 'programms parameter'


class GeodataUrlParameter(SandboxStringParameter):
    name = 'geodata_url'
    description = 'URL to geodata3.bin'
    default_value = "rsync://veles.yandex.ru/Berkanavt/geodata/geodata3.bin"


class BlockstatUrlParameter(SandboxStringParameter):
    name = 'blockstat_url'
    description = 'URL to blockstat.dict'
    default_value = "rsync://veles.yandex.ru/Berkanavt/statdata/blockstat.dict"


class BuildExpMRServer(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Experiments Adminka.
        Checkouts adminka from SVN and
        creates virtualenv with necessary packets.
    """
    type = 'BUILD_EXP_MR_SERVER'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [ArcadiaUrlParameter, AbtResourceSelector, SpuResourceSelector, OddResourceSelector, GeodataUrlParameter, BlockstatUrlParameter]

    URL_EXP_MR_SERVER_SVN = 'quality/ab_testing/scripts/exp_mr_server'

    PATH_PACKET = 'exp_mr_server'
    PATH_TGZ = 'exp_mr_server.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Experiments MR server {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * export adminka project
        * create virtualenv
        * install dependencies there
        * create tgz archive
        * create shard
        """

        import virtualenv

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "exp_mr_server")
        path_bin = os.path.join(path_checkout, "exp_mr_server", "bin")
        path_formaters = os.path.join(path_checkout, "exp_mr_server", "formaters")
        path_virtualenv = os.path.join(path_packet, "exp_mr_server", "env")

        self._export_arcadia(self.URL_EXP_MR_SERVER_SVN, path_checkout)
        os.mkdir(path_bin)

        self._extract_bins(path_bin, self.ctx[AbtResourceSelector.name], ["stat_collector", "stat_fetcher", "stat_long_fetcher"])
        self._extract_bins(path_bin, self.ctx[SpuResourceSelector.name], ["spu-test"])
        self._extract_bins(path_bin, self.ctx[OddResourceSelector.name], ["odd"])
        self._extract_bins(path_formaters, self.ctx[SpuResourceSelector.name], ["web.py"], path_packet_bin="Berkanavt/ab_testing/scripts/interface")

        url_geodata = self.ctx[GeodataUrlParameter.name]
        logging.info("Extract geodata {}".format(url_geodata))
        self._download_file(url_geodata, path_bin)

        url_blockstat = self.ctx[BlockstatUrlParameter.name]
        logging.info("Extract blockstat.dict {}".format(url_blockstat))
        self._download_file(url_blockstat, path_bin)

        logging.info("Path: {} {} {} {}".format(path_packet, path_checkout, path_virtualenv, path_bin))

        '''for resource_type, target, target_path in self.get_target_resources():
            if resource_type is not None and self._target_enabled(target):
                filename = os.path.basename(target_path)
                resource_path = os.path.join(self.LOCAL_BIN_DIR, filename)
                shutil.copy(resource_path, os.path.join(path_bin, filename))'''

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)

        self._install_dependencies(path_virtualenv, path_checkout)

        virtualenv.make_environment_relocatable(path_virtualenv)

        with open(os.path.join(path_checkout, "default_settings.py"), "a") as f:
            f.write("\nEXP_MR_SERVER_REVISION = '%d'\n\n" % (int(revision)))

        self._make_resource(path_checkout)
        self._create_shard()

    def _extract_bins(self, path_bin, id_res, bins, path_packet_bin="Berkanavt/ab_testing/bin"):
        tmp = self.path("tmp")

        remove_path(tmp)
        os.mkdir(tmp)

        res = channel.task.sync_resource(id_res)
        copy_path(res, os.path.join(tmp, "tmp.tar.gz"))

        run_process(['tar', 'xvzf', os.path.join(tmp, "tmp.tar.gz"), '-C', tmp])
        for deb in os.listdir(tmp):
            if deb[-4:] == ".deb":
                run_process(['dpkg', '-x', os.path.join(tmp, deb), tmp])
                for binary in bins:
                    copy_path(os.path.join(tmp, path_packet_bin, binary), os.path.join(path_bin, binary))

    def _install_dependencies(self, path_virtualenv, path_checkout):
        logging.info('install python libs')
        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')
        python_path = os.path.join(path_virtualenv, 'bin', 'python')

        run_process([python_path, pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', os.path.join(path_checkout, 'pip-requirements.txt')],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

    def _make_resource(self, path_packet):
        logging.info('create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='experiments exp_mr_server tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.EXP_MR_SERVER,
            arch='linux'
        )

    def _create_shard(self):
        shard_name = 'expmrserver-r{0}-{1}'.format(self.ctx['revision'], time.strftime("%Y%m%d_%H%M%S"))

        logging.info('create shard {0}'.format(shard_name))
        self.ctx['shard_name'] = shard_name

        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)

        shutil.copy(self.path(self.PATH_TGZ), shard_path)
        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_path, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.PATH_TGZ)
            )

        logging.info('create shard resource')
        self.create_resource(
            description='experiments mr server shard r{0}'.format(self.ctx['revision']),
            resource_path=shard_path,
            resource_type=resource_types.EXP_MR_SERVER_SHARD,
            arch='linux'
        )

        # do initialize shard
        logging.info('initialize shard')
        run_process(
            ['perl', utils.get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)

    def _download_file(self, url, path):
        logging.info("DOWNLOAD {} to {}".format(url, path))
        if url[:8] == "rsync://":
            run_process(
                ['rsync', url, path],
                log_prefix="rsync-download",
                work_dir=self.path()
            )


__Task__ = BuildExpMRServer
