# coding: utf-8

import os
import logging
import re
import shutil
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class AdminkaEnvParameter(LastReleasedResource):
    name = 'expadm_env'
    description = 'Virtual environment'
    resource_type = resource_types.EXPERIMENTS_ADMINKA_ENV


class AdminkaGeoParameter(LastReleasedResource):
    name = 'expadm_geo'
    description = 'geodata5.tsv'
    resource_type = resource_types.EXPERIMENTS_ADMINKA_GEO


class BuildExperimentsAdminka(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Experiments Adminka.
        Checkouts adminka from SVN and
        creates virtualenv with necessary packets.
    """
    type = 'BUILD_EXPERIMENTS_ADMINKA'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [ArcadiaUrlParameter,
                        AdminkaEnvParameter,
                        AdminkaGeoParameter]

    URL_ADMINKA_SVN = 'quality/ab_testing/scripts/adminka'

    URLS_ARCADIA_PACKAGES = [
        ('rem/client', 'remclient'),
        ('quality/userdata/rpc_api/userdata_api_client.py', 'userdata_api_client.py'),
        ('quality/userdata/rpc_api/api_lib/json_tools.py', 'json_tools.py'),
        ('quality/ab_testing/scripts/abwi_server/abwi_rpc_client.py', 'abwi_rpc_client.py'),
    ]

    PATH_PACKET = 'adminka'
    PATH_TGZ = 'experiments_adminka.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Experiments Adminka {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * export adminka project
        * create virtualenv
        * install dependencies there
        * create tgz archive
        * create shard
        """

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "adminka")
        path_virtualenv = os.path.join(path_packet, "env")

        os.mkdir(path_packet)
        self._export_arcadia(self.URL_ADMINKA_SVN, path_checkout)

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        self._create_virtualenv(path_virtualenv)

        self._validate_virtualenv(path_virtualenv, path_checkout)

        self._install_libs_from_arcadia(path_virtualenv)
        self._prepare_geobase(path_checkout)

        with open(os.path.join(path_checkout, "adminka/default_settings.py"), "a") as f:
            f.write("\nADMINKA_REVISION = '%d'\n\n" % (int(revision)))

        self._make_resource(path_packet)

    def _create_virtualenv(self, path_virtualenv):
        logging.info('extracting virtualenv')

        path_tar = channel.task.sync_resource(self.ctx[AdminkaEnvParameter.name])

        os.mkdir(path_virtualenv)

        with tarfile.open(path_tar, "r") as f:
            f.extractall(path=path_virtualenv)

    def _validate_virtualenv(self, path_virtualenv, path_checkout):
        logging.info('validating virtualenv')

        with open(os.path.join(path_virtualenv, "pip-requirements.txt"), "r") as f:
            reqs_virtualenv = f.read()
        with open(os.path.join(path_checkout, "pip-requirements.txt"), "r") as f:
            reqs_checkout = f.read()

        if not reqs_virtualenv == reqs_checkout:
            logging.error("OUTDATED virtualenv")
            raise Exception("Virtualenv is outdated, please build new using BUILD_EXPERIMENTS_ADMINKA_ENV task")

    def _install_libs_from_arcadia(self, path_virtualenv):
        logging.info('export python libs from Arcadia')
        path_packages = os.path.join(path_virtualenv, 'lib', 'python2.7', 'site-packages')

        for pack in self.URLS_ARCADIA_PACKAGES:
            self._export_arcadia(pack[0],
                                 os.path.join(path_packages, pack[1]))

    def _prepare_geobase(self, path_checkout):
        path_src = channel.task.sync_resource(self.ctx[AdminkaGeoParameter.name])
        path_dst = os.path.join(path_checkout, "adminka", "geodata5.tsv")

        shutil.copy(path_src, path_dst)

        return path_dst

    def _make_resource(self, path_packet):
        logging.info('create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            tar.dereference = True
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='experiments adminka tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.EXPERIMENTS_ADMINKA,
            arch='linux'
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildExperimentsAdminka
