# coding: utf-8

import os
import logging
import re
import shutil
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class BuildExperimentsAdminkaEnv(SandboxTask):
    """
        Build experiments adminka virtual environment.
    """
    type = 'BUILD_EXPERIMENTS_ADMINKA_ENV'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [ArcadiaUrlParameter]

    URL_REQUIREMENTS = 'quality/ab_testing/scripts/adminka/pip-requirements.txt'

    PATH_ENV = 'env'
    PATH_TGZ = 'expadm_env.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Expadm venv {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * create virtualenv
        * install dependencies there
        * create tgz archive
        """

        import virtualenv

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_virtualenv = self.path(self.PATH_ENV)
        path_req = self.path("pip-requirements.txt")

        logging.info('exporting requirements')
        self._export_arcadia(self.URL_REQUIREMENTS, path_req)

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)

        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')

        # https://github.com/pypa/setuptools/issues/942#issuecomment-274817619
        # pip upgrade must be separate to have up-to-date pip running following install
        logging.info('updating pip')
        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     'pip==9.0.3', ],  # USEREXP-5710
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

        logging.info('updating wheel and setuptools')
        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     'wheel',
                     'setuptools==38.6.0', ],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

        logging.info('install python libs')
        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     '-r', path_req],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

        logging.info('making env relocatable')
        virtualenv.make_environment_relocatable(path_virtualenv)

        logging.info('placing pip-requirements to env')
        shutil.copy(path_req, os.path.join(path_virtualenv, "pip-requirements.txt"))

        logging.info('creating tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            tar.dereference = True
            for entry in os.listdir(path_virtualenv):
                tar.add(os.path.join(path_virtualenv, entry), entry)

        logging.info('creating resource')
        self.create_resource(
            description='expadm env tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.EXPERIMENTS_ADMINKA_ENV,
            arch='linux'
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildExperimentsAdminkaEnv
