# coding: utf-8

import logging
import time
import urllib

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Svn
from sandbox.sandboxsdk.task import SandboxTask


class GeodataUrlParameter(SandboxStringParameter):
    name = 'geodata_url'
    description = 'URL to geodata5.bin'
    default_value = "rsync://veles.yandex.ru/Berkanavt/geodata/geodata5.bin"


class AdminkaUtilsGeoParameter(LastReleasedResource):
    name = 'utils_geo'
    description = 'Utility extract_geo_tree'
    resource_type = resource_types.EXPERIMENTS_ADMINKA_UTILS_GEO


class BuildExperimentsAdminkaGeo(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Prepare geo-tree for Experiments Adminka.
    """
    type = 'BUILD_EXPERIMENTS_ADMINKA_GEO'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [GeodataUrlParameter,
                        AdminkaUtilsGeoParameter]

    def on_execute(self):
        path_geodata = self.path("geodata5.bin")
        path_geodata_tsv = self.path("geodata5.tsv")

        url_geodata = self.ctx[GeodataUrlParameter.name]

        logging.info("Extract geodata {}".format(url_geodata))
        self.ctx['geodata_download_time'] = int(time.time())
        self._download_file(url_geodata, path_geodata)

        path_extracter = channel.task.sync_resource(self.ctx[AdminkaUtilsGeoParameter.name])

        run_process(
            [path_extracter,
             "-g", path_geodata,
             "-o", path_geodata_tsv],
            log_prefix="extract-geobase",
            work_dir=self.path()
        )

        logging.info("creating resource")
        self.create_resource(
            description='expadm geo {}'.format(self.ctx['geodata_download_time']),
            resource_path=path_geodata_tsv,
            resource_type=resource_types.EXPERIMENTS_ADMINKA_GEO,
            arch='linux'
        )

    def _download_file(self, url, path):
        logging.info("DOWNLOAD {} to {}".format(url, path))
        if url[:8] == "rsync://":
            run_process(
                ['rsync', url, path],
                log_prefix="rsync-download",
                work_dir=self.path()
            )
        elif url[:10] == "svn+ssh://":
            Svn.export(url, path)
        else:
            urllib.urlretrieve(url, path)


__Task__ = BuildExperimentsAdminkaGeo
