# -*- coding: utf-8 -*-
import logging
import os
import stat
import tarfile
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter

from sandbox.sandboxsdk.task import SandboxTask
import importlib
import inspect


class ResourcesCount(SandboxIntegerParameter):
    name = 'res_count'
    description = 'Number of resources'
    default_value = 1


class ResourceType(SandboxStringParameter):
    name = 'res_type'
    description = 'Resource type'
    default_value = 'YABS_YTSTAT_PERFORMANCE'


class ArchiveName(SandboxStringParameter):
    name = 'arch_name'
    description = 'Name of archive inside resource'


class BinariesCount(SandboxIntegerParameter):
    name = 'bin_count'
    description = 'Number of binary files'
    default_value = 1


class BinariesNames(SandboxStringParameter):
    name = 'bin_names'
    description = 'Name of binaries inside archive, ";" separated'


class BuildFakeResource(SandboxTask):
    type = 'BUILD_FAKE_RESOURCE'

    input_parameters = (
        ResourcesCount,
        ResourceType,
        ArchiveName,
        BinariesCount,
        BinariesNames,
    )

    def _make_executable(self, full_path):
        """
            Метод делает переданный путь исполняемым
        """
        st = os.stat(full_path)
        os.chmod(full_path, st.st_mode | stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH)

    def _create_fake_exe(self, path, name):
        """
            Метод создает фейковый бинарный файл
            делает его исполняемым
        """
        if path and not os.path.isdir(path):
            os.makedirs(path)
        full_path = os.path.join(path, name)
        with open(full_path, 'wb') as fout:
            fout.write(os.urandom(1024))
        logging.info("Creating fake exe {}".format(full_path))
        self._make_executable(full_path)

    def is_exe(self, fpath):
        """
            Метод проверяет -- является ли файл исполняемым
        """
        return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    def _get_resource_by_name(self, name):
        """
            Метод возвращает ресурс по заданному названию
        """
        module = importlib.import_module("sandbox.projects.resource_types")
        for curname, obj in inspect.getmembers(module):
            if inspect.isclass(obj) and name == curname:
                return obj
        return None

    def on_execute(self):
        """
            Создает бинарные файлы полученные из параметров
            Упаковывает их в архив
            Выкладывает в виде ресурса с типом полученным из параметра
        """
        res_count = int(self.ctx['res_count'])
        res_type = self.ctx['res_type']
        arch_name = self.ctx['arch_name']
        logging.info(os.listdir('.'))
        if os.path.exists(arch_name):
            os.remove(arch_name)
        bins_count = int(self.ctx['bin_count'])
        bins_names = self.ctx['bin_names'].split(';')
        if len(bins_names) != bins_count:
            logging.warn("Количество ресурсов не совпадает с требуемым")
        for bin_name in bins_names:
            self._create_fake_exe('', bin_name)
        for i in xrange(0, res_count):
            self._create_archive(arch_name + str(i + 1), bins_names)
        for i in xrange(0, res_count):
            self._build_resource(res_type, arch_name + str(i + 1) + '.tar.gz')

    def _build_resource(self, name_type, archive_path):
        """
            Метод билдит ресурс за заданным типом и кладет в него архив
        """
        self.create_resource("RESORUCE", archive_path, self._get_resource_by_name(name_type))

    def _create_archive(self, arch_name, binaries):
        """
            Создает архив в который укладывает бинарные файлы
        """
        with tarfile.open(arch_name + '.tar.gz', 'w:gz') as tar:
            for binary in binaries:
                tar.add(binary)


__Task__ = BuildFakeResource
