# coding: utf-8


import json
import logging
import os
import os.path
import re
import shutil
import sys
import tarfile
import time

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.BuildFilterTrie.resources import FILTER_TRIE_META

class FetchRemote(SandboxBoolParameter):
    name = 'fetch_flag'
    description = "Don't build data. Fetch from URLs"
    default_value = True


class ReleaseStatus(SandboxRadioParameter):
    choices = [(_, _) for _ in ("none", "prestable", "stable", "testing", "unstable")]
    description = '"stable" Will be immediately enqueued to production deploy, so use with care'
    default_value = "none"
    name = 'release_status'


class ReleaseUniqDataOnly(SandboxBoolParameter):
    name = 'release_uniq_data'
    description = "Release only if last released resource has different md5"
    default_value = True


class ResourceURL_FT(SandboxStringParameter):
    description = 'Url for filter.trie'
    default_value = 'rsync://sas1-1799.search.yandex.net:26492/filter/current/cat/flt/filter.trie'
    name = 'resource_url_ft'


class ResourceURL_MT(SandboxStringParameter):
    description = 'Url for mirrors.trie'
    default_value = 'rsync://sas1-1799.search.yandex.net:26492/filter/current/src/misc/mirrors.trie'
    name = 'resource_url_mt'


class SkipTest(SandboxBoolParameter):
    name = 'skip_test'
    description = "Don't test data. I DO KNOW what I'm doing"
    default_value = False


class BuildFilterTrie(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_FILTER_TRIE'

    description = 'filter.trie and mirrors.trie, SEPE-7389, SEPE-7325'
    execution_space = 16000

    input_parameters = [FetchRemote, ReleaseStatus, ReleaseUniqDataOnly, SkipTest, ResourceURL_FT, ResourceURL_MT]

    environment = (
        environments.PipEnvironment('yandex-yt'),
        environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
    )

    def test_data(self):
        raise Exception("Tests not implemented yet")

    def on_execute(self):
        if self.ctx['fetch_flag']:
            from yt.wrapper import YtClient, ypath_join
            from sandbox import sdk2, common

            url_ft = self.ctx['resource_url_ft']
            url_mt = self.ctx['resource_url_mt']
            try:
                token = sdk2.Vault.data("WEBMASTER", "robot-webmaster-yt-token")
            except common.errors.VaultError:
                raise common.errors.TaskFailure("Need secret to use")

            if re.compile("^rsync://").search(url_ft):
                self.remote_copy(self.ctx['resource_url_ft'], '.')
            else:
                parts = url_ft.split(':')
                with open("filter.trie", "wb") as fp:
                    fp.write(YtClient(parts[0], token).read_file(ypath_join(parts[1], "filter.trie")).read())

            if re.compile("^rsync://").search(url_mt):
                self.remote_copy(self.ctx['resource_url_mt'], '.')
            else:
                parts = url_mt.split(':')
                with open("mirrors.trie", "wb") as fp:
                    stream = YtClient(parts[0], token).read_file(ypath_join(parts[1], "mirrors.trie"))
                    buf = True
                    while buf:
                        buf = stream.read(length=1024768)
                        fp.write(buf)

        else:
#            os.system('touch filter.trie mirrors.trie') # FIXME: tests only
            raise Exception("Internal build not implemented yet")

        self.create_resource(
            arch='any',
            attributes={'ttl': 1},
            description='SEPE-7389, SEPE-7325',
            resource_path='filter.trie',
            resource_type=resource_types.FILTER_TRIE
        )
        self.create_resource(
            arch='any',
            attributes={'ttl': 1},
            description='SEPE-7389, SEPE-7325',
            resource_path='mirrors.trie',
            resource_type=resource_types.MIRRORS_TRIE
        )

        # write metadata file, SERP-41529
        dmeta = {"task": self.id, "mtime": int(time.time())}
        fmeta = open('metadata.json', "w")
        json.dump(dmeta, fmeta)
        fmeta.close()

        self.create_resource(
            arch='linux',
            attributes={'ttl': 1},
            description='Metadata for filter.trie and mirrors.trie, SEPE-14778',
            resource_path='metadata.json',
            resource_type=FILTER_TRIE_META
        )

        if self.ctx['skip_test'] is not True:
            self.test_data()
            logging.info('All tests passed')
        else:
            logging.warning("Flag 'skip_test' is set by releaser, tests skipped")

        if self.ctx['release_status'] is not 'none':
            check_already_released = False
            if self.ctx['release_uniq_data'] is True:
                check_already_released = str(resource_types.FILTER_TRIE) + " " + str(resource_types.MIRRORS_TRIE)

            releaser = self.create_subtask(
                task_type='RELEASE_ANY',
                inherit_notifications=True,
                input_parameters={
                    'check_already_released': check_already_released,
                    'release_task_id': self.id,
                    'release_status': self.ctx['release_status'],
                },
                description="{filter,mirrors}.trie (task id: " + str(self.id) + ' autorelease)'
            )

            self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        self.mark_released_resources(self.ctx['release_status'], ttl=180)


__Task__ = BuildFilterTrie
