# -*- coding: utf-8 -*-
"""
Build gperftools from source tarball
Author: mvel@
"""

import os
import shutil

from sandbox.projects import resource_types
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp


class GperftoolsTarball(sp.ResourceSelector):
    name = 'gperftools_tgz_resource_id'
    # obtained from https://github.com/gperftools/gperftools
    description = 'GPT source code tarball'
    resource_type = 'OTHER_RESOURCE'
    required = True


class BuildGperfTools(SandboxTask):
    """
    Executes GPT build sequence
    """
    type = 'BUILD_GPERFTOOLS'

    input_parameters = (
        GperftoolsTarball,
    )

    def on_execute(self):
        os.chdir(self.abs_path())

        gpt_gz = './gperftools.tar.gz'

        shutil.copy(
            self.sync_resource(self.ctx[GperftoolsTarball.name]),
            gpt_gz)

        process.run_process(
            ['tar', '-xvzf', gpt_gz],
            log_prefix='extract-gperftools',
        )
        process.run_process(['rm', gpt_gz])
        files = os.listdir('.')
        dir_name = None
        for fname in files:
            if 'gperftools-' in fname:
                dir_name = fname

        Arcadia.export(
            Arcadia.trunk_url('devtools/tools_build/build_gperftools.sh'),
            self.abs_path('build_gperftools.sh')
        )

        process.run_process(['chmod', '+x', 'build_gperftools.sh'])
        process.run_process(['./build_gperftools.sh'], log_prefix='make')

        self.create_resource(
            'GPT profiler library',
            dir_name + '/.libs/libprofiler.so',
            resource_types.GPERFTOOLS_PROFILER_SO_BINARY,
        )

        self.create_resource(
            'GPT pprof tool',
            dir_name + '/src/pprof',
            resource_types.GPERFTOOLS_PPROF_SCRIPT,
        )


__Task__ = BuildGperfTools
