# coding: utf-8

import distutils
import logging
import os

from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.build.parameters import ArcadiaPatch
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia


class Params:
    class BaseSvnUrl(parameters.SandboxArcadiaUrlParameter):
        name = "svn_url"
        description = "Svn url for branch with geotools"
        default_value = "arcadia:/arc/trunk"

    class YaMakeGeotools(parameters.SandboxBoolParameter):
        name = "ya_make_geotools"
        description = "Use ya make for build geosearch tools"
        default_value = False
        required = False

    params = (
        BaseSvnUrl,
        YaMakeGeotools,
        ArcadiaUrl,
        ArcadiaPatch,
    )


class BuildGeosearchTools(nanny.ReleaseToNannyTask, YaMakeTask):
    """
        Bundle with geosearch tools from arcadia
    """
    type = "BUILD_GEOSEARCH_TOOLS"

    input_parameters = Params.params

    def on_enqueue(self):
        self.ctx['build_system'] = 'ya'
        YaMakeTask.on_enqueue(self)
        channel.task = self
        self.create_resource(
            self.descr,
            "data",
            resource_types.GEOSEARCH_TOOLS,
            attributes={
                "ttl": 90,
            },
            arch="any",
        )

    def on_execute(self):
        data_dir = self.abs_path("data")
        if not os.path.isdir(data_dir):
            os.mkdir(data_dir)

        def export_tests_data_folder(svn_branch_url, folder):
            svn_url = svn_subfolder_url(svn_branch_url, folder)
            dst_dir = os.path.join(data_dir, folder)
            Arcadia.export(svn_url, dst_dir)
            with open(os.path.join(dst_dir, "version_info"), "w") as f:
                f.write("svn url: {}\n".format(svn_url))
                f.write("sandbox task: http://sandbox.yandex-team.ru/sandbox/tasks/view?task_id={}\n".format(self.id))

        self.data_dir = data_dir
        export_tests_data_folder(self.ctx[Params.BaseSvnUrl.name], "arcadia/search/geo/tools")
        if self.required_ya_make():
            self.ctx['targets'] = 'search/geo/tools'
            YaMakeTask.on_execute(self)

    def post_build(self, source_dir, output_dir, pack_dir):
        if self.required_ya_make():
            YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)
            export_ya_make_result(
                os.path.join(output_dir, 'search/geo/tools/'),
                self.data_dir,
            )

    def required_ya_make(self):
        return self.ctx.get(Params.YaMakeGeotools.name, False)


def svn_subfolder_url(svn_url, subfolder):
    svn_info = Arcadia.info(svn_url)
    return "{}/{}@{}".format(svn_info["url"], subfolder, svn_info["entry_revision"])


def export_ya_make_result(src_dir, dst_dir):
    logging.info('export_ya_make_result {} {}'.format(src_dir, dst_dir))
    distutils.dir_util.copy_tree(src_dir, dst_dir)  # use replace copytree version


__Task__ = BuildGeosearchTools
