# coding: utf-8

import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import ssh

from sandbox.projects.common.juggler import base

import sandbox.common.types.misc as ctm


class RepoNameParameter(parameters.SandboxStringParameter):
    name = 'repo_name'
    description = 'Repo name'
    default_value = 'sysmon-common'
    required = True


class BuildGolemDeb(base.BaseBuildJugglerDebTask):
    """
    Build golem deb package and upload it to dist.yandex.ru.
    """

    type = 'BUILD_GOLEM_DEB'

    execution_space = 5000

    input_parameters = base.BaseBuildJugglerTask.input_parameters + [
        RepoNameParameter,
        base.VersionPostfixParameter
    ]

    GIT_URL_TEMPLATE = "ssh://git@github.yandex-team.ru/monitoring/{name}.git"
    REPO_NAME = 'golem'

    privileged = True

    dns = ctm.DnsType.DNS64

    def _extract_deps(self):
        proc = process.run_process(
            ('dpkg-checkbuilddeps', ),
            outs_to_pipe=True, check=False,
            work_dir=self._checkout_path)
        _, stderr_data = proc.communicate()
        _, _, deps = stderr_data.strip().partition("Unmet build dependencies:")
        if deps:
            return deps.strip().split(" ")
        else:
            return []

    def _extract_last_version(self):
        proc = process.run_process(
            ('git', 'describe', '--tags', '--match=v*', '--abbrev=0', 'HEAD'),
            outs_to_pipe=True,
            work_dir=self._checkout_path)
        stdout_data, _ = proc.communicate()
        return tuple(int(x) for x in stdout_data.strip().lstrip('v').split('.'))

    def _replace_changelog(self):
        with open(os.path.join(self._checkout_path, 'debian', 'changelog'), 'w') as stream:
            process.run_process(
                (os.path.join(self._checkout_path, 'contrib', 'debian-changelog-from-git-log.sh'),),
                stdout=stream,
                work_dir=self._checkout_path,
                log_prefix='generate_changelog')

    def on_execute(self):
        apt_config = self.path('testing.list')
        with open(apt_config, 'w') as stream:
            stream.write('deb http://common.dist.yandex.ru/common/ testing/amd64/')
        process.run_process(('sudo', '-n', 'mv', apt_config, '/etc/apt/sources.list.d/testing.list'))

        apt_get = lambda *args: process.run_process(('sudo', '-n', 'apt-get', '-y') + args, log_prefix='apt_get')
        apt_get('update')
        apt_get('install', 'npm=1.3.4-1', 'nodejs=0.8.26-yandex1')

        deps = self._extract_deps()
        if deps:
            apt_get('install', *deps)

        version_tuple = self._extract_last_version()
        new_version = 'v{0}.{1}.{2}'.format(version_tuple[0], version_tuple[1], version_tuple[2] + 1)
        process.run_process(('git', 'tag', new_version), work_dir=self._checkout_path)
        self._replace_changelog()

        version_postfix = self.ctx.get(base.VersionPostfixParameter.name)
        if version_postfix:
            package_version = self._update_changelog(version_postfix)
        else:
            package_version = self._extract_changelog_version()
        self._build_deb(self.ctx.get(RepoNameParameter.name, RepoNameParameter.default_value),
                        version=package_version)
        self.ctx["changelog_last_comment"] = self._extract_changelog_comment()

        if not version_postfix:
            with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
                process.run_process(
                    ('git', 'push', 'origin', '--tags'),
                    work_dir=self._checkout_path,
                    log_prefix='push_tags')

    def on_release(self, additional_parameters):
        self._create_conductor_ticket(additional_parameters['release_status'])


__Task__ = BuildGolemDeb
