# -*- coding: utf-8 -*-
import logging
import shutil
import tarfile
from os import environ, listdir, path
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxGitUrlParameter, \
    SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from time import time

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny


class GitUrl(SandboxGitUrlParameter):
    default_value = 'https://git.qe-infra.yandex-team.ru/scm/searchmon/greenbox.git'
    required = True


class GitBranch(SandboxStringParameter):
    name = 'git_branch'
    description = 'GreenBox git branch'
    default_value = 'master'
    required = True


class PythonSource(LastReleasedResource):
    name = 'python_source'
    description = 'Python source'
    resource_type = resource_types.PYTHON_SOURCE
    required = False


class CompressedPythonVirtualenv(LastReleasedResource):
    name = 'python_virtualenv'
    description = 'Compressed python virtualenv'
    resource_type = resource_types.GREENBOX_VIRTUALENV
    required = False


class BuildGreenboxBundle(SandboxTask, nanny.ReleaseToNannyTask):
    """
        Build greenbox bundle (tar.gz, virtualenv with installed greenbox package).
        Virtualenv can be built either from python source or from existing virtualenv with 'virtualenv' module.
    """

    type = 'BUILD_GREENBOX_BUNDLE'
    input_parameters = [GitUrl, GitBranch, PythonSource, CompressedPythonVirtualenv]

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе
            Может быть переопределён в наследниках для уточнения возвращаемых данных

            :return список из трёх значений revision, tag, branch
        """
        return self.ctx.get('revision', None), self.ctx.get('tag', None), self.ctx.get('git_branch', None)

    def on_execute(self):

        if not self.ctx['python_source'] and not self.ctx['python_virtualenv']:
            raise RuntimeError('Either python_source or python_virtualenv must be defined!')

        if self.ctx['python_source'] and self.ctx['python_virtualenv']:
            raise RuntimeError('Both python_source and python_virtualenv are defined!')

        _timestamp = int(round(time()))
        self.ctx['revision'] = _timestamp
        tokenized_git_url = self.ctx['git_url']

        token = self.get_vault_data(self.owner, 'robot_maestro_stash_oauth')
        if tokenized_git_url.startswith('https://'):
            new = 'https://x-oauth-token:{}@'.format(token)
            tokenized_git_url = tokenized_git_url.replace('https://', new)
            logging.warning('git url was tokenized: {}'.format(
                tokenized_git_url.replace(token, '***')))

        logging.info('checking out greenbox source from {}'.format(tokenized_git_url))
        _greenbox_src = self.path('greenbox_src')
        run_process(
            ['git', 'clone', tokenized_git_url, _greenbox_src],
            log_prefix='git_clone', check=True, wait=True, environment={}
        )
        run_process(
            ['git', 'checkout', self.ctx['git_branch']],
            log_prefix='git_checkout', check=True, wait=True, work_dir=_greenbox_src, environment={}
        )

        if self.ctx['python_source']:

            #
            # Source part
            #

            logging.info('\n\nBUILDING PYTHON FROM SOURCE\n\n')

            logging.info('downloading python source')
            _python_source_resource_id = self.ctx['python_source']
            _python_source_resource = self.sync_resource(_python_source_resource_id)

            _python_source = self.path('python-source')
            logging.info('decompressing python source to {}'.format(_python_source))
            with tarfile.open(_python_source_resource) as _tar:
                _tar.extractall(path=_python_source)
                _source_prefix = _tar.getnames()[0]
                _python_version = _source_prefix.split('-')[1][:3]
                _python_source = path.join(_python_source, _source_prefix)
                logging.info('python source decompressed to {}'.format(_python_source))

            _python_dist = self.path('python-dist')
            logging.info('configuring python')
            run_process(
                ['./configure', '--prefix={}'.format(_python_dist)],
                log_prefix='python_configure', check=True, wait=True, work_dir=_python_source, environment={}
            )

            logging.info('building python')
            run_process(
                ['make'],
                log_prefix='python_make', check=True, wait=True, work_dir=_python_source, environment={}
            )

            logging.info('installing python to {}'.format(_python_dist))
            run_process(
                ['make', 'install'],
                log_prefix='python_make_install', check=True, wait=True, work_dir=_python_source, environment={}
            )
            _python_bin = path.join(_python_dist, 'bin', 'python{}'.format(_python_version))
            assert path.isfile(_python_bin)

            logging.info('installing virtualenv')
            run_process(
                [
                    _python_bin, '-m', 'pip', 'install',
                    '--only-binary', 'all', '-f', path.join(_greenbox_src, 'wheel'), 'virtualenv'
                ],
                log_prefix='pip_install_virtualenv', check=True, wait=True, environment={}
            )
            _virtualenv_bin = path.join(_python_dist, 'bin', 'virtualenv')

            _greenbox_env = self.path('greenbox-env')
            _python_bin = path.join(_python_dist, 'bin', 'python{}'.format(_python_version))

            logging.info('using python at {}'.format(_python_bin))
            assert path.isfile(_python_bin)

            logging.info('using virtualenv at {}'.format(_virtualenv_bin))
            assert path.isfile(_virtualenv_bin)

            logging.info('creating new virtualenv at {}'.format(_greenbox_env))
            run_process(
                [
                    _virtualenv_bin, '--always-copy', '-p', _python_bin, _greenbox_env
                ],
                log_prefix='virtualenv_create_new', check=True, wait=True,
                environment={
                    'PATH': '{}:{}'.format(path.join(_python_dist, 'bin'), environ['PATH'])
                }
            )

            _existing_lib = path.join(_greenbox_env, 'lib', 'python{}'.format(_python_version))
            logging.info('deleting existing lib: {}'.format(_existing_lib))
            shutil.rmtree(_existing_lib)

            logging.info('copying lib from {}'.format(_python_dist, 'lib', 'python{}'.format(_python_version)))
            shutil.copytree(
                path.join(_python_dist, 'lib', 'python{}'.format(_python_version)),
                path.join(_greenbox_env, 'lib', 'python{}'.format(_python_version))
            )

        else:

            #
            # Virtualenv part
            #

            logging.info('\n\nUSING PYTHON FROM VIRTUALENV\n\n')

            logging.info('downloading python virtualenv')
            _python_virtualenv_resource_id = self.ctx['python_virtualenv']
            _python_virtualenv_resource = self.sync_resource(_python_virtualenv_resource_id)

            _python_virtualenv = self.path('python-virtualenv')
            logging.info('decompressing virtualenv to {}'.format(_python_virtualenv))
            with tarfile.open(_python_virtualenv_resource) as _tar:
                _tar.extractall(path=_python_virtualenv)
                _python_dist = path.join(_python_virtualenv, listdir(_python_virtualenv)[0])
                logging.debug('uncompressed venv\n{}\n{}\n{}'.format(
                    listdir(_python_virtualenv), _python_dist, listdir(_python_dist)
                ))
                _python_version = listdir(path.join(_python_dist, 'lib'))[0].replace('python', '')
                logging.debug('python version is {}'.format(_python_version))

            _greenbox_env = _python_dist

        #
        # Common part
        #

        _greenbox_python_bin = path.join(_greenbox_env, 'bin', 'python')
        assert path.isfile(_greenbox_python_bin)

        _greenbox_lib_path = path.join(_greenbox_env, 'lib', 'python{}'.format(_python_version))
        assert path.isdir(_greenbox_lib_path)

        logging.info('installing greenbox')
        run_process(
            [
                _greenbox_python_bin, '-m', 'pip', 'install',
                '--no-index',
                '--find-links', path.join(_greenbox_src, 'wheel'),
                '--verbose',
                '--prefix', _greenbox_env,
                _greenbox_src
            ],
            log_prefix='pip_install_greenbox', check=True, wait=True,
            environment={
                'PYTHONPATH': path.join(_greenbox_lib_path, 'site-packages')
            }
        )

        logging.info('making virtualenv relocatable')
        run_process(
            [_greenbox_python_bin, '-m', 'virtualenv', '--relocatable', _greenbox_env],
            log_prefix='virtualenv_make_relocatable', check=True, wait=True,
            environment={
                'PYTHONPATH': path.join(_greenbox_lib_path, 'site-packages')
            }
        )

        logging.info('compressing venv')
        _tar_path = self.path('greenbox.tar')
        with tarfile.TarFile(_tar_path, 'w') as _tar:
            _tar.add(_greenbox_env, 'greenbox')

        logging.info('creating resource')
        self.create_resource(
            description=self.ctx.get(
                'description', 'Greenbox tar, built from {} at {}'.format(self.ctx['git_branch'], _timestamp)
            ),
            resource_path=_tar_path,
            resource_type=resource_types.GREENBOX_BUNDLE,
            arch='any'
        )

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildGreenboxBundle
