# coding: utf-8

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.parameters import SandboxStringParameter
import os
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.task as sdk_task


class HAProxySourceVersion(SandboxStringParameter):
    name = 'haproxy_source_version'
    description = 'Version of HAProxy'
    default_value = '1.7.8'


class LibreSslSourceVersion(SandboxStringParameter):
    name = 'libressl_source_version'
    description = 'Version of LibreSSL'
    default_value = '2.4.5'


class PcreSourceVersion(SandboxStringParameter):
    name = 'pcre_source_version'
    description = 'Version of PCRE'
    default_value = '8.41'


class BuildHAProxy(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    '''
    Build HAProxy statically from sources.
    '''

    type = 'BUILD_HAPROXY'
    client_tags = Tag.LINUX_PRECISE
    dns = ctm.DnsType.DNS64
    input_parameters = [
        HAProxySourceVersion,
        LibreSslSourceVersion,
        PcreSourceVersion
    ]

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    def on_execute(self):
        haproxy_source_ver = self.ctx['haproxy_source_version']
        haproxy_version_family = '.'.join(haproxy_source_ver.split('.')[:2])
        haproxy_source_url = 'http://www.haproxy.org/download/%s/src/haproxy-%s.tar.gz'

        libressl_source_url = 'http://ftp.openbsd.org/pub/OpenBSD/LibreSSL/libressl-%s.tar.gz'
        libressl_source_ver = self.ctx['libressl_source_version']

        pcre_source_url = 'ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-%s.tar.gz'
        pcre_source_ver = self.ctx['pcre_source_version']

        current_dir = os.path.abspath('./')
        libressl_dir = os.path.join(current_dir, 'libressl-%s' % libressl_source_ver)
        pcre_dir = os.path.join(current_dir, 'pcre-%s' % pcre_source_ver)
        haproxy_dir = os.path.join(current_dir, 'haproxy-%s' % haproxy_source_ver)
        temporary_dir = os.path.join(current_dir, 'tmp')
        staticlibssl_dir = os.path.join(temporary_dir, 'staticlibssl')
        staticlibpcre_dir = os.path.join(temporary_dir, 'staticlibpcre')

        sdk_process.run_process(
            ['wget', '-q', libressl_source_url % libressl_source_ver],
            log_prefix='get_libressl'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'libressl-%s.tar.gz' % libressl_source_ver],
            log_prefix='tar_libressl'
        )

        sdk_process.run_process(
            ['wget', '-q', pcre_source_url % pcre_source_ver],
            log_prefix='get_pcre'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'pcre-%s.tar.gz' % pcre_source_ver],
            log_prefix='tar_pcre'
        )

        sdk_process.run_process(
            ['wget', '-q', haproxy_source_url % (haproxy_version_family, haproxy_source_ver)],
            log_prefix='get_haproxy'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'haproxy-%s.tar.gz' % haproxy_source_ver],
            log_prefix='tar_haproxy'
        )

        os.chdir(libressl_dir)

        sdk_process.run_process(
            [
                './configure',
                '--prefix=%s' % staticlibssl_dir,
                '--enable-shared=no'
            ],
            work_dir=libressl_dir,
            log_prefix='configure_libressl'
        )

        sdk_process.run_process(
            ['make'],
            work_dir=libressl_dir,
            log_prefix='make_libressl'
        )

        sdk_process.run_process(
            ['make', 'install'],
            work_dir=libressl_dir,
            log_prefix='make_libressl'
        )

        os.chdir(pcre_dir)

        sdk_process.run_process(
            [
                './configure',
                '--prefix=%s' % staticlibpcre_dir,
                '--enable-shared=no',
                '--enable-utf8',
                '--enable-jit'
            ],
            work_dir=pcre_dir,
            log_prefix='configure_pcre'
        )

        sdk_process.run_process(
            ['make'],
            work_dir=pcre_dir,
            log_prefix='make_pcre'
        )

        sdk_process.run_process(
            ['make', 'install'],
            work_dir=pcre_dir,
            log_prefix='make_pcre'
        )

        os.chdir(haproxy_dir)

        sdk_process.run_process(
            [
                'make',
                'TARGET=linux2628',
                'USE_PCRE_JIT=1',
                'USE_STATIC_PCRE=1',
                'USE_OPENSSL=1',
                'PCRE_LIB=%s/lib' % staticlibpcre_dir,
                'PCRE_INC=%s/include' % staticlibpcre_dir,
                'SSL_INC=%s/include' % staticlibssl_dir,
                'SSL_LIB=%s/lib' % staticlibssl_dir,
                'ADDLIB=-ldl -lrt',
                'USE_ZLIB=1',
                'USE_GETADDRINFO=1',
                'ARCH=x86_64',
                'USE_TFO=1',
                'USE_FUTEX=1',
            ],
            work_dir=haproxy_dir,
            log_prefix='make_haproxy'
        )

        self.create_resource(
            arch='linux',
            description=self.descr,
            resource_path='%s/haproxy' % haproxy_dir,
            resource_type='HAPROXY_EXECUTABLE',
        )


__Task__ = BuildHAProxy
