# coding: utf-8
import os
import logging
import tarfile

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = ''
    required = False


class BuildHttpGridfs(SandboxTask):
    type = 'BUILD_HTTP_GRIDFS'

    input_parameters = [GitTagParameter]

    GIT_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/nanny/http-gridfs.git'
    CHECKOUT_PATH = 'http_gridfs'
    TGZ_PATH = 'http_gridfs.tar.gz'

    client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_XENIAL

    def _get_checkout_url(self):
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    def _checkout(self, url):
        tag = self.ctx.get('tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'
        run_process(
            ['git', 'clone', url, self.CHECKOUT_PATH],
            log_prefix='git_clone',
            shell=True,
        )
        run_process(
            ['git', 'checkout', tag],
            work_dir=self.CHECKOUT_PATH,
            log_prefix='git_co',
            shell=True
        )

    def _install_package(self, virtualenv_path):
        pip_path = os.path.join(virtualenv_path, 'bin', 'pip')
        # update pip at first
        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     '-U', 'pip==18.0'],
                    log_prefix='http_gridfs_pip_update',
                    work_dir=self.CHECKOUT_PATH)

        # install dependencies into virtualenv
        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     '-r', 'pip-requirements.txt',
                     '-f', 'wheels'],
                    log_prefix='http_gridfs_reqs_pip_install',
                    work_dir=self.CHECKOUT_PATH)
        # now install project itself
        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     '.'],
                    log_prefix='pip_install',
                    work_dir=self.CHECKOUT_PATH)

    def _make_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description='http-gridfs virtualenv tgz {0}'.format(self.ctx['tag']),
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.HTTP_GRIDFS,
            arch='linux'
        )

    def _set_project_version(self):
        version_file_path = os.path.join('src', 'http_gridfs', 'version.py')
        run_process(['sed', '-i',
                     '"s/development_version/$(/skynet/python/bin/python setup.py --version)/"', version_file_path],
                    shell=True,
                    work_dir=self.CHECKOUT_PATH)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def on_execute(self):
        """
        Plan is:
        * git clone and checkout specified tag
        * create virtualenv
        * install project there
        * copy config
        * create tgz with bundle
        * create shard
        """
        checkout_url = self._get_checkout_url()
        self._checkout(checkout_url)
        virtualenv_path = self.path('build')
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', '--system-site-packages', virtualenv_path])

        self._set_project_version()
        self._install_package(virtualenv_path)

        run_process(['/skynet/python/bin/virtualenv', '--relocatable', virtualenv_path])

        self._make_resource(virtualenv_path)


__Task__ = BuildHttpGridfs
