# coding: utf-8

import datetime
import json
import logging
import os
import re
import tarfile
import time

from collections import OrderedDict

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask


class YtServer(SandboxStringParameter):
    name = 'yt_server'
    description = 'YT server'
    default_value = 'hahn'


class YtTable(SandboxStringParameter):
    name = 'yt_table'
    description = 'YT table'
    default_value = '//home/abt/icookie_blacklist/YYYYMMDD'


class VaultGroup(SandboxStringParameter):
    '''
    Vault group of yt token.
    '''
    name = 'vault_group'
    description = 'Vault group'
    default_value = 'AB-TESTING'


class VaultName(SandboxStringParameter):
    '''
    Vault name of yt token.
    '''
    name = 'vault_name'
    description = 'Vault name'
    default_value = 'yt-token'


class BuildIcookieBlacklist(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Icookie Blacklist.
    """
    type = 'BUILD_ICOOKIE_BLACKLIST'

    input_parameters = [
        YtServer,
        YtTable,
        VaultGroup,
        VaultName,
    ]

    environment = (
        environments.PipEnvironment('yandex-yt', '0.7.34-0'),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
    )

    PATH_TGZ = 'icookie_blacklist.tar.gz'

    MAX_BLACKLIST_SIZE = 10000000

    def on_execute(self):

        yt_server = self.ctx[YtServer.name]
        yt_table = self.ctx[YtTable.name]

        if not yt_server:
            raise SandboxTaskFailureError('Invalid yt_server name')
        if not yt_table:
            raise SandboxTaskFailureError('Invalid yt_table name')

        cookies = self._download_cookies(yt_server, yt_table)

        blacklist_size = len(cookies)
        self.ctx['blacklist_size'] = blacklist_size

        logging.info('Blacklist size = {}'.format(blacklist_size))

        path_packet = self.path('icookie_blacklist')
        path_blacklist = os.path.join(path_packet, 'icookie_blacklist.txt')
        path_metafile = os.path.join(path_packet, 'info.json')

        os.mkdir(path_packet)

        self._write_blacklist(path_blacklist, cookies)
        self._write_metafile(path_metafile, yt_server, yt_table, blacklist_size)

        self._make_resource(path_packet)

    def _download_cookies(self, yt_server, yt_table):
        logging.info('Download cookies from "{}":"{}"'.format(yt_server, yt_table))

        cookies = []

        yt_token = self.get_vault_data(
            utils.get_or_default(self.ctx, VaultGroup),
            utils.get_or_default(self.ctx, VaultName),
        )

        import yt.wrapper as yt
        yt_client = yt.YtClient(proxy=yt_server, token=yt_token)

        for row in yt_client.read_table(yt_table):
            cookie = row['icookie']

            if not re.search('^[A-Za-z0-9]{1,100}$', cookie):
                # very dumb check
                raise SandboxTaskFailureError('Invalid icookie value: "{}" at row {}'.format(cookie, len(cookies) + 1))

            cookies.append(cookie)

            if len(cookies) > self.MAX_BLACKLIST_SIZE:
                raise SandboxTaskFailureError('Row count exceeds {}'.format(self.MAX_BLACKLIST_SIZE))

        return cookies

    def _write_blacklist(self, path_blacklist, cookies):
        logging.info('Write blacklist into "{}"'.format(path_blacklist))

        with open(path_blacklist, 'w') as f:
            print >>f, len(cookies)

            for cookie in cookies:
                print >>f, cookie

    def _write_metafile(self, path_metafile, yt_server, yt_table, blacklist_size):
        logging.info('Write metadata into "{}"'.format(path_metafile))

        data = OrderedDict([
            ('time', int(time.time())),
            ('time_iso', datetime.datetime.now().isoformat()),
            ('blacklist_size', blacklist_size),
            ('yt_server', yt_server),
            ('yt_table', yt_table),
            ('task_author', self.author),
        ])

        text = json.dumps(data, indent=2)

        with open(path_metafile, 'w') as f:
            f.write(text)

    def _make_resource(self, path_packet):
        path_tgz = self.path(self.PATH_TGZ)

        logging.info('Create tgz file: "{}" --> "{}"'.format(path_packet, path_tgz))

        with tarfile.open(path_tgz, 'w:gz') as tar:
            tar.dereference = True
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)

        logging.info('Create resource')

        self.create_resource(
            description='icookie blacklist',
            resource_path=path_tgz,
            resource_type=resource_types.ICOOKIE_BLACKLIST_ARCHIVE,
            arch='linux'
        )


__Task__ = BuildIcookieBlacklist
