# -*- coding: utf-8 -*-
import os

from sandbox.projects import resource_types
from sandbox.projects.balancer import resources as balancer_resources
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import branch_tag_from_svn_url
from sandbox.projects.common.BaseBinariesAndConfigsTask import BaseBinariesAndConfigsTask
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.BuildConfigGenerator import GeneratorGitTagParameter
from sandbox.projects.BuildConfigGenerator import SVN_COMMIT_KEY


class ConfigArchiveResourceParameter(LastReleasedResource):
    name = 'config_archive_resource_id'
    description = 'Config archive'
    resource_type = resource_types.CONFIG_GENERATOR_GENERATED


GENERATOR_GIT_KEY = 'generator_git_commit_tag'


def generate_project_ctx_key(project):
    """
        Поле в контексте, показывающее, нужно ли генерить конфиги для данного проекта
    """
    return 'generate_%s' % project


def generate_project_parameter(settings_cls):
    """
        Создает bool-параметр по классу настроек одного из проектов improxy
    """
    class GenerateParameter(SandboxBoolParameter):
        name = generate_project_ctx_key(settings_cls.NAME)
        description = 'Generate %s configs' % settings_cls.NAME

    return GenerateParameter


class BuildImproxyBundle(nanny.ReleaseToNannyTask, BaseBinariesAndConfigsTask):
    type = 'BUILD_IMPROXY_BUNDLE'

    input_parameters = BaseBinariesAndConfigsTask.input_parameters +\
                       [ConfigArchiveResourceParameter] +\
                       [generate_project_parameter(cls) for cls in media_settings.ALL_SETTINGS_CLASSES]

    TARGET_RESOURCE_TYPES = (
        balancer_resources.BALANCER_IMPROXY_EXECUTABLE,
    )

    def arcadia_info(self):
        def get_generator_info(key_name):
            commit, tag = self.ctx[key_name]
            return commit, tag, None

        def merge_infos(first, second, separator):
            def merge_values(value1, value2):
                if value1 is None or value2 is None:
                    return None
                return value1 + separator + value2

            revision = first[0] + separator + second[0]
            branch = merge_values(first[2], second[2])
            tag = merge_values(first[1], second[1])
            # Если одна сущность создана из ветки, а другая из тега, то зарелизить не получится, т.к. везде будут None
            # Кажется, это не очень важно, т.к. из веток релизим редко.
            return revision, tag, branch

        generator_info = get_generator_info(GENERATOR_GIT_KEY)

        binary_svn_url = self.ctx[consts.ARCADIA_URL_KEY]
        binary_revision = Arcadia.parse_url(binary_svn_url).revision
        binary_branch, binary_tag = branch_tag_from_svn_url(binary_svn_url)

        binary_info = (binary_revision, binary_tag, binary_branch)
        if not self.is_build_configs_task():
            return binary_info
        elif not self.is_build_binaries_task():
            return generator_info
        else:
            return merge_infos(binary_info, generator_info, ',')

    def get_configs_dir(self):
        return self.abs_path('configs')

    def get_config_basename(self, name):
        return "improxy_" + name + ".cfg"

    def get_config_resource_path(self, name):
        """
            Путь к сгенеренному файлу конфига, который станет ресурсом
            basename совпадает с именем файла, выдаваемого генерилкой
        """
        return os.path.join(self.get_configs_dir(), self.get_config_basename(name))

    def get_configs_list(self):
        """
            Список имен генеримых конфигов
        """
        for cls in media_settings.ALL_SETTINGS_CLASSES:
            if self.ctx[generate_project_ctx_key(cls.NAME)]:
                for region in cls.IMPROXY_CONFIG_LOCATIONS:
                    yield "%s_%s" % (cls.NAME, region)

    def on_enqueue(self):
        BaseBinariesAndConfigsTask.on_enqueue(self)
        if self.is_build_configs_task():
            configs_list = list(self.get_configs_list())
            if not configs_list:
                raise SandboxTaskFailureError("No configs were selected for generation")
            for name in configs_list:
                self._create_resource(self.descr + ' ' + name, self.get_config_resource_path(name),
                                      media_settings.config_name2resource_type('improxy_' + name), arch='any')

    def on_execute(self):
        BaseBinariesAndConfigsTask.on_execute(self)

        def fill_generator_git_key(key_name, generator_resource_id):
            generator_task_id = channel.sandbox.get_resource(generator_resource_id).task_id
            generator_task = channel.sandbox.get_task(generator_task_id)
            self.ctx[key_name] = (generator_task.ctx[SVN_COMMIT_KEY],
                                  generator_task.ctx[GeneratorGitTagParameter.name])

        fill_generator_git_key(GENERATOR_GIT_KEY, self.ctx[ConfigArchiveResourceParameter.name])

        if self.is_build_configs_task():
            def extract_archive(archive_resource_id, parent_dir):
                archive_path = self.sync_resource(archive_resource_id)
                run_process(['tar', '-xf', archive_path], log_prefix='tar', work_dir=parent_dir)
                return os.path.join(parent_dir)

            make_folder(self.get_configs_dir(), True)
            generated_dir = extract_archive(self.ctx[ConfigArchiveResourceParameter.name], self.abs_path())
            output_subdir = 'generated/improxy'
            for name in self.get_configs_list():
                copy_path(os.path.join(generated_dir, output_subdir, self.get_config_basename(name)),
                          self.get_config_resource_path(name))


__Task__ = BuildImproxyBundle
