# coding: utf-8

# $Header$

from datetime import datetime

import logging

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask


class ReleaseStatus(SandboxRadioParameter):
    choices = [(_, _) for _ in ("none", "prestable", "stable", "testing", "unstable")]
    description = '"stable" Will be immediately enqueued to production deploy, so use with care'
    default_value = "none"
    name = 'release_status'


class ReleaseUniqDataOnly(SandboxBoolParameter):
    name = 'release_uniq_data'
    description = "Release only if last released resource has different md5"
    default_value = True


class ResourceURL(SandboxStringParameter):
    description = 'Url for infected_serp_masks'
    default_value = 'rsync://vdb.yandex.ru/infection_info/infected_serp_masks'
    name = 'resource_url'


class SkipTest(SandboxBoolParameter):
    name = 'skip_test'
    description = "Don't test data. I DO KNOW what I'm doing"
    default_value = False


class BuildInfectedSerpMasks(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_INFECTED_SERP_MASKS'

    description = 'infected_serp_masks, SEPE-7377, SEPE-7325'
    execution_space = 512

    input_parameters = [ReleaseStatus, ReleaseUniqDataOnly, ResourceURL, SkipTest]

    def test_data(self):
        file = open('infected_serp_masks', 'r')
        cnt = 0
        for line in file:
            cnt += 1
            line = line.rstrip("\n")

            # must contain exactly 5 fields
            items = line.split("\t")
            if len(items) != 5:
                raise SandboxTaskFailureError("Line " + str(cnt) + " has " + str(len(items)) + " fileds (5 expected)")

            # second and last fields must be something like "2016-04-15 11:13:46"
            try:
                datetime.strptime(items[1], '%Y-%m-%d %H:%M:%S')
                datetime.strptime(items[4], '%Y-%m-%d %H:%M:%S')
            except:
                raise SandboxTaskFailureError("Line " + str(cnt) + " has unsupported datetimes in 1 and/or 4 fields")

        file.close()

    def on_execute(self):
        self.remote_copy(self.ctx['resource_url'], '.')

        self.create_resource(
            arch='any',
            attributes={'ttl': 1},
            description='SEPE-7377, SEPE-7325',
            resource_path='infected_serp_masks',
            resource_type=resource_types.INFECTED_SERP_MASKS
        )

        if self.ctx['skip_test'] is not True:
            self.test_data()
            logging.info('All tests passed')
        else:
            logging.warning("Flag 'skip_test' is set by releaser, tests skipped")

        if self.ctx['release_status'] is not 'none':
            check_already_released = False
            if self.ctx['release_uniq_data'] is True:
                check_already_released = str(resource_types.INFECTED_SERP_MASKS)

            releaser = self.create_subtask(
                task_type='RELEASE_ANY',
                inherit_notifications=True,
                input_parameters={
                    'check_already_released': check_already_released,
                    'release_task_id': self.id,
                    'release_status': self.ctx['release_status'],
                },
                description="infected_serp_masks (task id: " + str(self.id) + ') autorelease'
            )

            self.info = "Subtask {} runned, waiting it's decision about release.\n\n".format(releaser.id)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        self.mark_released_resources(self.ctx['release_status'], ttl=180)


__Task__ = BuildInfectedSerpMasks
