# coding: utf-8
import os
import logging

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import constants
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const
from sandbox.projects.BuildInstanceCtlBinary import BuildInstanceCtlBinary

from sandbox.common.types import misc

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import copy_path


class GitRefIdParameter(SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class ReleaseParameter(SandboxBoolParameter):
    name = 'release'
    description = 'Create shard and resources after running the tests'
    default_value = False


class RevisionParameter(parameters.SandboxIntegerParameter):
    name = 'revision'
    description = 'Arcadia revision number'
    required = True


class PatchUrlParameter(parameters.SandboxStringParameter):
    name = 'patch_url'
    description = 'Patch URL'
    required = True


class SshdResourceId(parameters.SandboxStringParameter):
    name = 'sshd_resource_id'
    description = 'Sshd resource id'
    required = True


class JugglerClientResourceId(parameters.SandboxStringParameter):
    name = 'juggler_client_resource_id'
    description = 'juggler-client resource id'
    required = True


class GdbToolkitResourceId(parameters.SandboxStringParameter):
    name = 'gdb_toolkit_resource_id'
    description = 'gdb_toolkit resource id'
    required = True


class LogrotateBinaryResourceId(parameters.SandboxStringParameter):
    name = 'logrotate_binary_resource_id'
    description = 'logrotate binary resource id'
    required = True


class IssHookStatusBinaryResourceId(parameters.SandboxStringParameter):
    name = 'iss_hook_status_binary_resource_id'
    description = 'iss hook status binary resource id'
    required = True


class INSTANCECTL_JUGGLER_CLIENT_BINARY(resource_types.AbstractResource):
    """
        juggler-client for runtime cloud instances
    """
    releasable = True
    any_arch = True
    executable = True
    auto_backup = True
    releasers = ['nekto0n', 'alximik', 'frolstas', 'alonger', 'romanovich', 'disafonov', 'reddi', u'nanny-robot',
                 'ferenets', 'i-dyachkov']


class INSTANCECTL_LOGROTATE_BINARY(resource_types.AbstractResource):
    """
        logrotate binary for runtime cloud instances
    """
    releasable = True
    any_arch = True
    executable = True
    auto_backup = True
    releasers = ['nekto0n', 'alximik', 'frolstas', 'alonger', 'romanovich', 'disafonov', 'reddi', u'nanny-robot',
                 'ferenets', 'i-dyachkov']


class INSTANCECTL_GDB_TOOLKIT(resource_types.AbstractResource):
    """
        gdb layer for coredumps
    """
    releasable = True
    any_arch = True
    executable = True
    auto_backup = True
    releasers = ['nekto0n', 'alximik', 'frolstas', 'alonger', 'romanovich', 'disafonov', 'reddi', u'nanny-robot',
                 'ferenets', 'i-dyachkov']


class BuildInstanceCtl(nanny.ReleaseToNannyTask, SandboxTask):

    qloud_init_ssh_resource_type = 'QLOUD_INIT_SSH'

    type = 'BUILD_INSTANCE_CTL'

    ARCADIA_INSTANCE_CTL_PATH = 'infra/nanny/instancectl'
    ARCADIA_BUILT_INSTANCE_CTL_PATH = 'infra/nanny/instancectl/bin/instancectl'
    ARCADIA_INSTANCE_CTL_ISS_HOOKS_PATH = 'arcadia:/arc/trunk/arcadia/infra/nanny/instancectl/iss_hooks'

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        ReleaseParameter,
        PatchUrlParameter,
        nanny.StartrekTicketIdsParameter,
        SshdResourceId,
        JugglerClientResourceId,
        LogrotateBinaryResourceId,
        IssHookStatusBinaryResourceId,
        GdbToolkitResourceId,
    ]

    def arcadia_info(self):
        """
        Получение информации о задаче при релизе

        :return список из трёх значений revision, tag, branch
        """
        return None, self.ctx.get('tag'), None

    def create_hooks_resources(self, hooks_path):
        """
        Создать ресурсы для скриптов хуков
        """
        logging.debug('Get hooks from %s', hooks_path)
        hooks = {
            'iss_hook_start': resource_types.ISS_HOOK_START,
            'iss_hook_stop': resource_types.ISS_HOOK_STOP,
            'iss_hook_install': resource_types.ISS_HOOK_INSTALL,
            'iss_hook_reopenlogs': resource_types.ISS_HOOK_REOPENLOG,
            'iss_hook_uninstall': resource_types.ISS_HOOK_UNINSTALL,
            'iss_hook_notify': resource_types.ISS_HOOK_NOTIFY,
        }
        for hook_filename, hook_resource_type in hooks.iteritems():

            hook_path = os.path.join(hooks_path, hook_filename)
            if os.path.exists(hook_path):
                logging.info('Create %s hook resource', hook_filename)
                hook_resource_path = self.path(hook_filename)
                copy_path(hook_path, hook_resource_path)
                self.create_resource(
                    description='InstanceCtl {}'.format(self.get_version()),
                    resource_path=hook_resource_path,
                    resource_type=hook_resource_type,
                )
            else:
                logging.error('Cannot find %s hook file', hook_filename)

    def create_additional_resource(self, resource_id, file_name, resource_type):
        resource = channel.channel.sandbox.get_resource(resource_id)
        source_path = self.sync_resource(resource)
        destination_path = self.path(file_name)
        paths.copy_path(source_path, destination_path)
        self.create_resource(
            description='{} from resource: {}'.format(file_name, resource_id),
            resource_path=destination_path,
            resource_type=resource_type,
        )

    def get_revision(self):
        return self.ctx['ref_sha']

    def get_version(self):
        return self.ctx['ref_id']

    def build_instance_ctl(self):
        """
        Builds InstanceCtl binary in a subtask and downloads to the current host

        :return: path to build instancectl binary
        :type: unicode
        """
        sub_tasks = self.list_subtasks(load=True)

        if len(sub_tasks) == 0:
            sub_task_context = {
                'targets': self.ARCADIA_INSTANCE_CTL_PATH,
                'arts': self.ARCADIA_BUILT_INSTANCE_CTL_PATH,
                'result_rt': 'INSTANCECTL',
                'result_single_file': True,
                constants.DISABLE_TEST_TIMEOUT: True,
                constants.ARCADIA_URL_KEY: 'arcadia:/arc/trunk/arcadia@{}'.format(self.get_revision()),
                constants.TESTS_REQUESTED: True,
                constants.TEST_THREADS: 8,
                constants.TEST_TAG: 'ya:manual',
                constants.TARGET_PLATFORM_FLAGS: ('--target-platform DEFAULT-LINUX-X86_64 '
                                                  '--target-platform-flag MUSL=yes'),
                constants.ARCADIA_PATCH_KEY: self.ctx['patch_url'],
                constants.ENV_VARS: "NANNY_VAULT_OAUTH_TOKEN='$(vault:value:NANNY:robot_nanny_vault_nanny_oauth_token)' NANNY_TVM_CLIENT_SECRET='$(vault:value:NANNY:robot_nanny_tvm_client_secret)'",
                constants.RAM_DRIVE_SIZE: 8192,
                constants.USE_AAPI_FUSE: True,
                constants.ALLOW_AAPI_FALLBACK: True,
                constants.MUSL: True,
                constants.SANDBOX_TAGS: 'PORTOD',
            }
            # InstanceCtl must be built on the porto hosts and it must be subclass
            # of YaMake task, so build it with other task
            self.create_subtask(task_type=BuildInstanceCtlBinary.type,
                                description='instancectl: {}'.format(self.ctx['ref_id']),
                                input_parameters=sub_task_context)

        # Wait for task execution finish, fail on errors
        utils.check_subtasks_fails()

        build_instance_ctl_task = sub_tasks[0]

        resources = channel.channel.sandbox.list_resources(resource_type=resource_types.INSTANCECTL,
                                                           omit_failed=True,
                                                           task_id=build_instance_ctl_task.id)
        if resources is None or len(resources) != 1:
            raise SandboxTaskFailureError('Cannot get {} resource of task {}'.format(resource_types.INSTANCECTL,
                                                                                     build_instance_ctl_task.id))
        instance_ctl_resource = resources[0]

        source_path = self.sync_resource(instance_ctl_resource)
        destination_path = self.path('instancectl')

        paths.copy_path(source_path, destination_path)
        return destination_path

    def on_execute(self):
        """
        Plan is:
        * Build InstanceCtl via separate task which is subclass of YaMake
        * Get iss hooks content from the repo and create hoook resources
        """
        instance_ctl_path = self.build_instance_ctl()

        r = self.create_resource(description='InstanceCtl {}'.format(self.get_version()),
                                 resource_path=instance_ctl_path,
                                 resource_type=resource_types.INSTANCECTL,
                                 arch=misc.OSFamily.LINUX)
        utils.set_resource_attributes(r.id,
                                      {'version': self.get_version()})

        self.create_additional_resource(self.ctx[SshdResourceId.name],
                                        file_name='qloud-init-ssh',
                                        resource_type=resource_types.INSTANCECTL_SSHD)
        self.create_additional_resource(self.ctx[JugglerClientResourceId.name],
                                        file_name='juggler-client',
                                        resource_type=INSTANCECTL_JUGGLER_CLIENT_BINARY)
        self.create_additional_resource(self.ctx[LogrotateBinaryResourceId.name],
                                        file_name='logrotate',
                                        resource_type=INSTANCECTL_LOGROTATE_BINARY)
        self.create_additional_resource(self.ctx[IssHookStatusBinaryResourceId.name],
                                        file_name='iss_hook_status',
                                        resource_type=resource_types.ISS_HOOK_STATUS)
        self.create_additional_resource(self.ctx[GdbToolkitResourceId.name],
                                        file_name='gdb.tgz',
                                        resource_type=INSTANCECTL_GDB_TOOLKIT)
        hooks_dir = svn.Arcadia.get_arcadia_src_dir(url=self.ARCADIA_INSTANCE_CTL_ISS_HOOKS_PATH)
        self.create_hooks_resources(hooks_dir)

    def post_instancectl_release(self, nanny_client, nanny_api_url, data):
        r_id = nanny_client.create_instancectl_release(data)
        if r_id is None:
            self.set_info('Nanny InstanceCtl release request has been created successfully, '
                          'but its ID cannot be retrieved')
            return
        u = const.RELEASE_REQUEST_TEMPLATE.format(nanny_api_url=nanny_api_url, release_request_id=r_id)
        self.set_info('Nanny InstanceCtl release request <a href="{}">{}</a> was created'.format(u, r_id),
                      do_escape=False)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        resources = channel.channel.sandbox.list_resources(resource_type=resource_types.INSTANCECTL,
                                                           omit_failed=True,
                                                           task_id=self.id)
        if resources is None or len(resources) != 1:
            raise SandboxTaskFailureError('Cannot get {} resource of task {}'.format(resource_types.INSTANCECTL,
                                                                                     self.id))
        resource = resources[0]
        data = {
            "spec": {
                "title": additional_parameters.get('release_subject', ''),
                "desc": additional_parameters.get('release_comments', ''),
                "type": "INSTANCECTL_RELEASE",
                "instancectlRelease": {
                    "releaseType": additional_parameters['release_status'],
                    "instancectl": {
                        "version": self.get_version(),
                        "url": [resource.skynet_id],
                        "fetchableMeta": {
                            "type": "SANDBOX_RESOURCE",
                            "sandboxResource": {
                                "resourceType": str(resource.type),
                                "resourceId": str(resource.id),
                                "taskId": str(self.id),
                                "taskType": str(self.type)
                            }
                        }
                    }
                }
            }
        }
        self.post_instancectl_release(self.nanny_client, const.NANNY_API_URL, data)
        dev_nanny_api_url = const.NANNY_DEV_API_URL
        dev_nanny_oauth_token = self.get_vault_data('NANNY', 'nanny_robot_dev_nanny_oauth_token')
        dev_nanny_client = nanny.NannyClient(api_url=dev_nanny_api_url,
                                             oauth_token=dev_nanny_oauth_token)
        try:
            self.post_instancectl_release(dev_nanny_client, dev_nanny_api_url, data)
        except Exception as e:
            self.set_info('Could not create InstanceCtl release in dev-nanny: {}'.format(e))


__Task__ = BuildInstanceCtl
