# coding: utf-8
import json
import logging
import os
import subprocess
import urllib2

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.environments import SandboxMavenEnvironment, SandboxJavaJdkEnvironment
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk import ssh


from sandbox import common


class SandboxThriftCompilerEnvironment(SandboxEnvironment):
    """
        Окружение thrift
    """
    resource_type = 'THRIFT_COMPILER_EXECUTABLE'

    name = 'thrift'

    sys_path_utils = [name, ]

    def prepare(self):
        logging.info('Prepare Git environment, version %s' % self.version)
        env_folder = self.get_environment_folder()
        target_path = os.path.join(env_folder, self.name)
        if not os.path.exists(target_path):
            resource_path = self.get_environment_resource()
            copy_path(resource_path, target_path)
        os.environ['PATH'] = env_folder + ':' + os.environ['PATH']


API_URL = "https://bb.yandex-team.ru/rest/api/1.0/projects/SEARCH_INFRA/repos/iss/{}?start=0&limit=25"
API_TIMEOUT_SECONDS = 5
GIT_TAG_KEY = 'git_tag'
GIT_BRANCH_KEY = 'git_branch'
GIT_REF_KEY = 'git_ref'


def get_via_api(entity_type, sort_args):
    tags_json = urllib2.urlopen(API_URL.format(entity_type), timeout=API_TIMEOUT_SECONDS).read()
    tags_list = json.loads(tags_json)['values']
    return sorted([str(value['displayId']) for value in tags_list], **sort_args)


def get_tags():
    return get_via_api('tags', {'reverse': True})


def get_branches():
    # Sort by number of '/' first to put develop/master on top
    return get_via_api('branches', {})


class GitRefTypes(object):
    REF = 'ref'
    BRANCH = 'branch'
    TAG = 'tag'
    ALL = [REF, BRANCH, TAG]


class GitRefParameter(SandboxStringParameter):
    name = GIT_REF_KEY
    description = 'Any ref'
    default_value = ''
    required = True


class GitRefTypeParameter(SandboxStringParameter):
    name = 'git_ref_type'
    description = 'Git reference type'
    choices = [(x, x) for x in GitRefTypes.ALL]
    sub_fields = {
        GitRefTypes.TAG: [GIT_TAG_KEY],
        GitRefTypes.BRANCH: [GIT_BRANCH_KEY],
        GitRefTypes.REF: [GIT_REF_KEY],
    }
    required = True
    default_value = GitRefTypes.TAG


def get_choices(choices_getter, description, default_value=None):
    try:
        choices = choices_getter()
        return [(x, x) for x in choices], choices[0] if default_value is None else default_value
    except Exception as e:
        logging.info("Exception retrieving {} from stash: {}".format(description, e))
        return None, "Could not retrieve from stash, refresh or enter manually"


def create_tag_parameter():
    class GitTagParameter(SandboxStringParameter):
        name = GIT_TAG_KEY
        description = 'Tag'
        required = True

        @common.utils.singleton_classproperty
        def choice_data(cls):
            return get_choices(get_tags, "tags")

        @common.utils.singleton_classproperty
        def choices(cls):
            return cls.choice_data[0]

        @common.utils.singleton_classproperty
        def default_value(cls):
            return cls.choice_data[1]

    return GitTagParameter


def create_branch_parameter():
    class GitBranchParameter(SandboxStringParameter):
        name = GIT_BRANCH_KEY
        description = 'Branch'
        required = True

        @common.utils.singleton_classproperty
        def choice_data(cls):
            return get_choices(get_branches, "branches", "master")

        @common.utils.singleton_classproperty
        def choices(cls):
            return cls.choice_data[0]

        @common.utils.singleton_classproperty
        def default_value(cls):
            return cls.choice_data[1]

    return GitBranchParameter


class BuildIssShards(SandboxTask):
    """
        Задача сборки iss_shards
    """
    type = 'BUILD_ISS_SHARDS'
    dns = ctm.DnsType.DNS64  # We must download from external mavens
    client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY  # Run on linux because we do not have thrift compiler for other arches
    input_parameters = [
        GitRefTypeParameter,
        create_tag_parameter(),
        create_branch_parameter(),
        GitRefParameter
    ]

    environment = (
        SandboxJavaJdkEnvironment('1.8.0'),
        SandboxMavenEnvironment('3.2.2'),
        SandboxThriftCompilerEnvironment('0.9.2')
    )

    execution_space = 5000

    GIT_URL = 'ssh://git@bb.yandex-team.ru/search_infra/iss.git'
    CHECKOUT_DIR = 'iss'
    SHARD_TRACKER_API_PROJECT = 'shard-tracker-parent/shard-tracker-api'
    VAULT_OWNER = 'ISS'
    SSH_KEY_VAULT_NAME = 'robot-verter-secret'

    def _root_path(self):
        return self.abs_path(self.CHECKOUT_DIR)

    def _iss_shards_dir(self):
        return os.path.join(self._root_path(), 'iss-python/iss_shards')

    def _iss_shards_file(self):
        return os.path.join(self._iss_shards_dir(), 'iss_shards')

    def _get_git_ref(self):
        ref_type = self.ctx.get(GitRefTypeParameter.name)
        if ref_type == GitRefTypes.TAG:
            return self.ctx.get(GIT_TAG_KEY)
        elif ref_type == GitRefTypes.BRANCH:
            return self.ctx.get(GIT_BRANCH_KEY)
        elif ref_type == GitRefTypes.REF:
            return self.ctx.get(GIT_REF_KEY)
        else:
            raise SandboxTaskFailureError("Unknown ref type {}".format(ref_type))

    def _checkout(self):
        run_process(
            ['git', 'clone', self.GIT_URL, self._root_path()],
            log_prefix='git_clone',
        )
        specified_ref = self._get_git_ref()
        resolved_ref = run_process(
            ['git', 'rev-list', '-n', '1', specified_ref],
            work_dir=self._root_path(),
            stdout=subprocess.PIPE
        ).stdout.read()
        self.set_info('Ref {}, resolved to commit {}'.format(specified_ref, resolved_ref))
        run_process(
            ['git', 'checkout', specified_ref],
            work_dir=self._root_path(),
            log_prefix='git_checkout',
        )

    def _build(self):
        run_process(
            ['make'],
            work_dir=self._iss_shards_dir(),
            log_prefix='make',
        )

    def _make_resource(self):
        self.create_resource(
            description=self.descr,
            resource_path=self._iss_shards_file(),
            resource_type=resource_types.ISS_SHARDS,
        )

    def arcadia_info(self):
        return None, self._get_git_ref(), None

    def on_execute(self):
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            self._checkout()
        self._build()
        self._make_resource()


__Task__ = BuildIssShards
