# coding: utf-8
import os
import logging
import tarfile

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.environments import SwatMongoDbEnvironment, PortablePyPyEnvironment
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny.nanny import StartrekTicketIdsParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process


class GitRefIdParameter(SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class ReleaseParameter(SandboxBoolParameter):
    name = 'release'
    description = 'Create resources after tests'
    default_value = False


class PypyVersion(SandboxStringParameter):
    name = 'pypy_version'
    description = 'Version of pypy portable'
    default_value = '5.6'
    required = False


class BuildIts(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_ITS'

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        ReleaseParameter,
        PypyVersion,
        StartrekTicketIdsParameter
    ]

    environment = [
        SwatMongoDbEnvironment('3.4.2'),
        PortablePyPyEnvironment('5.6'),
    ]

    GIT_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/nanny/its.git'
    CHECKOUT_PATH = 'its'
    TGZ_PATH = 'its.tar.gz'
    PIP_TGZ = 'pip-9.0.1.tar.gz'
    BUILD_DIR = 'build_dir'

    def _get_checkout_url(self):
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    def _checkout(self, url, dest, ref=None):
        run_process(
            ['git', 'clone', url, dest],
            log_prefix='git_clone',
            shell=True,
        )
        if ref is not None:
            run_process(
                ['git', 'checkout', ref],
                work_dir=dest,
                log_prefix='git_checkout',
                shell=True)

    client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE

    def _install_package(self, pip_path, wheels_dir='./wheels', requirements_txt='pip-requirements.txt'):
        its_src_dir = self.CHECKOUT_PATH
        # update pip to be able to install wheels
        run_process([pip_path, 'install', '-U', self.PIP_TGZ],
                    work_dir=its_src_dir)
        # install dependencies from local dir, fail if some are not satisfied
        run_process([pip_path, 'install', '--no-index',
                     '--find-links={}'.format(wheels_dir),
                     '-r', requirements_txt],
                    log_prefix='its_reqs_pip_install',
                    work_dir=its_src_dir)
        # now install ITS itself
        run_process([pip_path, 'install', '--no-index', '.'],
                    log_prefix='its_pip_install',
                    work_dir=its_src_dir)

    def _make_resource(self, virtualenv_path, description):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description=description,
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.ITS,
            arch='linux'
        )

    def _set_its_version(self):
        version_file_path = os.path.join('src', 'its', 'version.py')
        run_process(['sed', '-i',
                     '"s/development_version/$(/skynet/python/bin/python setup.py --version)/"', version_file_path],
                    shell=True,
                    work_dir=self.CHECKOUT_PATH)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def _run_tests(self, venv_path, wheels_dir='./wheels', requirements_txt='pip-requirements-dev.txt'):
        venv_bin_path = os.path.join(venv_path, 'bin')
        pip_path = os.path.join(venv_bin_path, 'pip')

        # install dependencies into virtualenv
        run_process([pip_path, 'install', '--no-index',
                     '--find-links={}'.format(wheels_dir),
                     '-r', requirements_txt],
                    log_prefix='its_dev_reqs_pip_install',
                    work_dir=self.CHECKOUT_PATH)

        env = dict(os.environ)
        path_parts = [venv_bin_path]
        if 'PATH' in env:
            path_parts.append(env['PATH'])
        env_path = ':'.join(path_parts)
        env.update({'PATH': env_path})

        old_do_not_restart = self.ctx['do_not_restart']
        self.ctx['do_not_restart'] = True
        run_process(['./test.sh'],
                    environment=env,
                    log_prefix='test_sh',
                    work_dir=self.CHECKOUT_PATH)
        self.ctx['do_not_restart'] = old_do_not_restart

    def on_execute(self):
        logging.info('checking out the source code')
        ref_id = self.ctx['ref_id']
        ref_sha = self.ctx.get('ref_sha')

        make_folder(self.path(self.BUILD_DIR))

        checkout_url = self._get_checkout_url()
        self._checkout(checkout_url, self.CHECKOUT_PATH, ref=ref_sha or ref_id)

        venv_path = self.path('its_build')

        pypy_version = self.ctx.get('pypy_version')
        if pypy_version:
            pypy_root_path = self._prepare_pypy()
            python_bin_path = os.path.join(pypy_root_path, 'bin')
            virtualenv_bin = os.path.join(python_bin_path, 'virtualenv-pypy')
            run_process([virtualenv_bin, '--no-site-packages', '--always-copy', venv_path])
        else:
            python_bin_path = '/skynet/python/bin/'
            virtualenv_bin = os.path.join(python_bin_path, 'virtualenv')
            run_process([virtualenv_bin, '--system-site-packages', venv_path])

        venv_bin_path = os.path.join(venv_path, 'bin')
        pip_path = os.path.join(venv_bin_path, 'pip')

        self._set_its_version()
        if pypy_version:
            self._install_package(pip_path, wheels_dir='./pypy_wheels', requirements_txt='pypy-pip-requirements.txt')
        else:
            self._install_package(pip_path)

        logging.info('run tests')
        if pypy_version:
            self._run_tests(venv_path, wheels_dir='./pypy_wheels')
        else:
            self._run_tests(venv_path)

        if pypy_version:
            run_process(['cp', '--remove-destination',
                         os.path.join(pypy_root_path, 'bin/*'),
                         os.path.join(venv_path, 'bin')],
                        shell=True)
            run_process(['cp', '-r',
                         os.path.join(pypy_root_path, 'lib_pypy'),
                         venv_path])
            run_process(['cp', '-r', '--no-clobber',
                         os.path.join(pypy_root_path, 'lib-python'),
                         venv_path])

        if self.ctx.get('release'):
            run_process([virtualenv_bin, '--relocatable', venv_path])
            if ref_id.startswith('refs/tags/'):
                tag = ref_id[len('refs/tags/'):]
            else:
                tag = 'master'
            description = 'its virtualenv tgz {0}'.format(tag)
            self._make_resource(venv_path, description)

    def _prepare_pypy(self):
        pypy_root_path = os.path.join(self.path(self.BUILD_DIR), 'pypy')
        copy_path(os.environ['PORTABLE_PYPY_DIR'], pypy_root_path)

        pypy_bin = os.path.join(pypy_root_path, 'bin/pypy')
        logging.info('Installing pip...')
        run_process([pypy_bin, '-m', 'ensurepip'], log_prefix='install_pip',
                    work_dir=pypy_root_path)
        return pypy_root_path


__Task__ = BuildIts
