import logging
import os
from collections import defaultdict
from datetime import datetime

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask
from sandbox.projects.common.utils import get_or_default
from sandbox.projects import resource_types
import sandbox.projects.BuildIznankaCompanyUrlsTrie.resources as iznanka_resources
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE
from sandbox.sandboxsdk.sandboxapi import RELEASE_TESTING
from sandbox.projects.common.solomon import upload_to_solomon
import sandbox.projects.common.file_utils as fu


class Params(object):
    ''' Input parameters '''
    class UrlsExtractor(sdk_parameters.ResourceSelector):
        name = 'urls_extractor'
        description = 'Urls extractor'
        resource_type = iznanka_resources.IznankaUrlsExtractorExecutable

    class TrieCompiler(sdk_parameters.ResourceSelector):
        name = 'trie_compiler'
        description = 'Trie compiler'
        resource_type = resource_types.TRIECOMPILER_EXECUTABLE

    class ReleaseAttr(sdk_parameters.SandboxStringParameter):
        name = 'release_attr'
        description = 'Where to release'
        choices = [
            ('testing', RELEASE_TESTING),
            ('stable', RELEASE_STABLE),
            ('don\'t release', ''),
        ]
        default_value = ''

    class StatsEnvironment(sdk_parameters.SandboxStringParameter):
        name = 'stats_environment'
        description = 'StatsEnvironment'
        choices = [
            ('testing', 'testing'),
            ('stable', 'stable'),
            ('none', 'none'),
        ]
        default_value = 'none'

    Content = [
        UrlsExtractor,
        TrieCompiler,
        ReleaseAttr,
        StatsEnvironment,
    ]


class Builder(ReleaseToNannyTask, sdk_task.SandboxTask, Params):
    ''' Iznanka company urls trie builder '''

    type = 'IZNANKA_COMPANY_URLS_BUILDER'
    input_parameters = Params.Content

    environment = [sdk_environments.PipEnvironment('yandex-yt')]

    def create_sensors(self, stats):
        sensors = []
        timestamp = int(datetime.now().strftime('%s'))
        for k, v in stats.items():
            sensors.append({
                'labels': {
                    'sensor': str(k),
                },
                'ts': timestamp,
                'value': float(v),
            })
        return sensors

    def on_execute(self):
        urls_extractor = self.sync_resource(self.ctx.get(self.UrlsExtractor.name))
        trie_compiler = self.sync_resource(self.ctx.get(self.TrieCompiler.name))

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = self.get_vault_data(self.owner, 'yt-token')
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        iznanka_urls_file = 'iznanka_urls'

        extract_cmd = [urls_extractor,
                       '--dst', iznanka_urls_file,
                       ]

        logging.info('Run command: {}'.format(' '.join(extract_cmd)))
        run_process(extract_cmd, stderr=open(os.path.join(get_logs_folder(), 'urls_extractor.stderr'), 'w'))
        logging.info('Done')

        iznanka_urls_trie_file = 'iznanka_urls.trie'

        build_trie_cmd = ['cat',
                          iznanka_urls_file,
                          '|', trie_compiler,
                          '-t', 'ui16',
                          iznanka_urls_trie_file,
                          ]

        logging.info('Run command: {}'.format(' '.join(build_trie_cmd)))
        # run_process(build_trie_cmd, stderr=open(os.path.join(get_logs_folder(), 'trie_compiler.stderr'), 'w'))
        run_process(' '.join(build_trie_cmd), shell=True, log_prefix="trie_compiler")

        logging.info('Done')

        self.create_resource('iznanka urls tsv', iznanka_urls_file, iznanka_resources.IznankaCompanyUrlsTsv)
        self.create_resource('iznanka urls trie', iznanka_urls_trie_file, iznanka_resources.IznankaCompanyUrlsTrie)

        stats = defaultdict(int)
        for line in fu.read_line_by_line(iznanka_urls_file):
            name, status = line.split('\t')
            stats[status] += 1

        if get_or_default(self.ctx, self.StatsEnvironment) != 'none':
            solomon_common_labels = {
                'project': 'iznanka_company_urls_builder',
                'cluster': 'main',
                'service': self.ctx.get(self.StatsEnvironment.name),
            }
            upload_to_solomon(solomon_common_labels, self.create_sensors(stats))

        release_type = get_or_default(self.ctx, self.ReleaseAttr)
        if release_type:
            self.create_subtask(
                task_type='RELEASE_ANY',
                inherit_notifications=True,
                input_parameters={
                    'release_task_id': self.id,
                    'release_status': release_type,
                },
                description=('backa POI companies and urls (task id: {}) autorelease' .format(self.id))
            )


__Task__ = Builder
