# coding: utf-8


import os
import platform
import os.path
import time

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import copy_path

from sandbox.projects.common.utils import get_bsconfig
import urllib2
import zipfile


class BuildJavaJDK(SandboxTask):
    '''
    Share java archive as resource and shard
    '''
    type = 'BUILD_JAVA_JDK'

    execution_space = 1000

    java_version = '8u60'
    #java_version = '8u45'
    java_version_string = '1.8.0_60'
    #java_source_url = 'http://download.oracle.com/otn-pub/java/jdk/%s-b14/jdk-%s-linux-x64.tar.gz' % (java_version, java_version)
    java_source_url = 'http://download.oracle.com/otn-pub/java/jdk/%s-b27/jdk-%s-linux-x64.tar.gz' % (java_version, java_version)
    java_tzupdater_version = "2.0.0"
    java_tzupdater_tzdata_version = "2015a"
    java_tzupdater_url = "http://download.oracle.com/otn-pub/java/tzupdater/%s/tzupdater-%s-%s.zip" % (java_tzupdater_version,
                                                                                                        java_tzupdater_version.replace(".", "_"),
                                                                                                        java_tzupdater_tzdata_version
                                                                                                    )

    def arcadia_info(self):
        """
        Gets info for release
        """
        return self.java_tzupdater_tzdata_version, 'java', self.java_version

    # Helper to download java and tzupdater
    def _download_from_oracle(self, url, local_filename):
        url_handle = urllib2.Request(url)

        # We should download from Oracle website using special cookie, that means we accept license agreement
        url_handle.add_header("Cookie", "oraclelicense=accept-securebackup-cookie")
        target_url = urllib2.urlopen(url_handle)

        with open(local_filename, "w") as file_handler:
            file_handler.writelines(target_url)
            file_handler.close()

    def on_execute(self):

        java_shard_name = 'java_bundle-%s-%s-%s' % (self.java_version, self.java_tzupdater_tzdata_version, time.strftime("%s"))
        self.ctx['shard_name'] = java_shard_name

        os.mkdir(java_shard_name)
        os.chdir(self.abs_path(''))

        java_tar_gz_path = self.abs_path('java.tgz')
        java_tzupdater_zip_path = self.abs_path('tzupdater.zip')
        java_extracted_path = self.abs_path('jdk%s' % self.java_version_string)
        java_extracted_path_short = 'jdk%s' % self.java_version_string
        java_tzupdater_path = self.abs_path('tzupdater-%s-%s/tzupdater.jar' % (self.java_tzupdater_version, self.java_tzupdater_tzdata_version))

        shard_path = self.path(java_shard_name)

        self._download_from_oracle(self.java_source_url, java_tar_gz_path)
        self._download_from_oracle(self.java_tzupdater_url, java_tzupdater_zip_path)

        run_process(['tar', '-xzf', java_tar_gz_path],  shell=True, log_prefix='tar-out')

        zipfile.ZipFile(java_tzupdater_zip_path).extractall("./")

        run_process([
                            'JAVA_HOME=%s' % java_extracted_path,
                            '%s/bin/java' % java_extracted_path, '-jar',
                            java_tzupdater_path, '--update', '--force'
                        ],
                        shell=True, log_prefix='tzdata-update')

        run_process([
                            'JAVA_HOME=%s' % java_extracted_path,
                            '%s/bin/java' % java_extracted_path, '-jar',
                            java_tzupdater_path, '--version'
                        ],
                        shell=True, log_prefix='tzdata-version')

        run_process(['tar', '-czf', self.abs_path('new-java.tgz'), java_extracted_path_short],
                        shell=True, log_prefix='tar-in')

        with open(os.path.join(java_shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar --strip-components=1 -xzf {}\n'.format('java.tgz')
            )

        copy_path(self.abs_path('new-java.tgz'), "%s/java.tgz" % shard_path)
        copy_path(self.abs_path('new-java.tgz'), java_tar_gz_path)

        self.create_resource(
            description=java_shard_name,
            resource_path=shard_path,
            resource_type=resource_types.JAVA_SHARD,
            attributes={
                "version": '%s' % self.java_version,
                "ttl": "inf",
                "platform": platform.platform()
            }
        )

        self.create_resource(
            description="java bundle",
            resource_path=java_tar_gz_path,
            resource_type=resource_types.JAVA_ARCHIVE,
            attributes={
                "version": '%s' % self.java_version,
                "ttl": "inf",
                "platform": platform.platform()
            }
        )

        # do initialize shard
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', java_shard_name],
            log_prefix="bsconfig",
            work_dir=self.path()
        )


__Task__ = BuildJavaJDK
