# -*- coding: utf-8 -*-

import os

from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

# from projects.common import apihelpers
from sandbox.projects.common import utils
from sandbox.projects import resource_types


class Tier(sp.SandboxStringParameter):
    name = "tier"
    description = "Tier"
    required = True
    default_value = None


class Group(sp.SandboxStringParameter):
    name = "group"
    description = "Primus group"
    required = True
    default_value = None


class Server(sp.SandboxStringParameter):
    name = "server"
    description = "Map reduce cluster name"
    required = True
    default_value = "banach.yt.yandex.net"


class YtPath(sp.SandboxStringParameter):
    name = "path"
    description = "Path to jupiter yt prefix"
    required = True
    default_value = "//home/jupiter"


class Generation(sp.SandboxStringParameter):
    name = "generation"
    description = "Database generation (YT_STATE)"
    required = True
    default_value = None


class JupiterBundle(sp.LastReleasedResource):
    name = "jupiter_bundle_resource_id"
    description = "Jupiter bundle (will use last released if omitted)"
    required = False


class VaultGroup(sp.SandboxStringParameter):
    '''
    Vault group of yt token.
    '''
    name = 'vault_group'
    description = 'Vault group'
    default_value = 'AB-TESTING'


class VaultName(sp.SandboxStringParameter):
    '''
    Vault name of yt token.
    '''
    name = 'vault_name'
    description = 'Vault name'
    default_value = 'yt-token'


OUT_SHARD_RESOURCE_ID = "out_shard_resource_id"


class BuildJupiterShard(SandboxTask):
    """
        Build shard in Sandbox using Jupiter technology
    """
    type = "BUILD_JUPITER_SHARD"
    input_parameters = (
        Tier,
        Group,
        Generation,
        JupiterBundle,
        Server,
        YtPath,
        VaultGroup,
        VaultName
    )

    execution_space = 240000
    required_ram = 85 * 1024
    client_tags = ~Tag.LXC

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx[OUT_SHARD_RESOURCE_ID] = self.create_resource(
            '{tier}-{group} shard built from {generation}'.format(
                tier=utils.get_or_default(self.ctx, Tier),
                group=utils.get_or_default(self.ctx, Group),
                generation=utils.get_or_default(self.ctx, Generation),
            ),
            'primus-{tier}-{group}-gen{generation}'.format(
                tier=utils.get_or_default(self.ctx, Tier),
                group=utils.get_or_default(self.ctx, Group),
                generation=utils.get_or_default(self.ctx, Generation),
            ),
            resource_types.SEARCH_DATABASE,
        ).id

    def on_execute(self):
        # create folder for shard
        shard_path = channel.sandbox.get_resource(self.ctx[OUT_SHARD_RESOURCE_ID]).path
        paths.make_folder(shard_path, delete_content=True)
        # obtain jupiter bundle
        jupiter_bundle_resource_id = utils.get_or_default(self.ctx, JupiterBundle)
        bundle_path = self.sync_resource(jupiter_bundle_resource_id)  # 167719857)
        run_sh = os.path.join(bundle_path, 'run.sh')
        # prepare welding params
        tier = utils.get_or_default(self.ctx, Tier)
        group = utils.get_or_default(self.ctx, Group)
        generation = utils.get_or_default(self.ctx, Generation)
        mr_server = utils.get_or_default(self.ctx, Server)
        yt_path = utils.get_or_default(self.ctx, YtPath)
        yt_token = self.get_vault_data(
            utils.get_or_default(self.ctx, VaultGroup),
            utils.get_or_default(self.ctx, VaultName),
        )

        os.environ['YT_LOG_LEVEL'] = 'DEBUG'
        if yt_token:
            os.environ['YT_TOKEN'] = yt_token
        # weld!
        process.run_process(
            [
                '/bin/bash',
                run_sh,
                '-t', tier,
                '-s', group,
                '-g', generation,
                '-o', shard_path,
                '-h', mr_server,
                '-p', yt_path
            ],
            shell=True,
            log_prefix='weld'
        )


__Task__ = BuildJupiterShard
