# -*- coding: utf-8 -*-

import copy
import glob
import json
import os
import re
import logging
import shutil
import sys
import threading

from sandbox.common.types.client import Tag

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.projects.common.utils import get_svn_info
from sandbox.projects import resource_types
from sandbox.projects.common.commands import get_make_name
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.projects.BuildKiwiTriggers import branch_cfg
from sandbox.projects.BuildKiwiTriggers import cfg
from sandbox.projects.BuildKiwiTriggers import deploy
from sandbox.projects.BuildKiwiTriggers import metaquery
from sandbox.projects.BuildKiwiTriggers import pbmsg
from sandbox.projects.BuildKiwiTriggers import util
from sandbox.projects.BuildKiwiTriggers.data_cache import DataCache
from sandbox.projects.BuildKiwiTriggers.data_scheme import SchemeCp
from sandbox.projects.BuildKiwiTriggers.post_cp_hooks import PostCpHook
from sandbox.projects.BuildKiwiTriggers.triggers_data import PathEval
from sandbox.projects.BuildKiwiTriggers.triggers_data import TriggersData
# from projects.common import rrt_parameters
from sandbox.projects.BuildKiwiTriggers import form

from sandbox.projects.common import apihelpers


class BuildKiwiTriggers(ArcadiaTask):
    # TODO: Document the class
    type = 'BUILD_KIWI_TRIGGERS'

    client_tags = Tag.LINUX_PRECISE
    execution_space = 75 * 1024  # 75Gb
    BUILD_DIR = 'build'
    RESULT_FOLDER = 'kiwi_queries'

    CONTEXT_FILE = 'ya_kiwi_build.context.json'
    SYMLINKS_DIR = 'symlinks'
    YA_SUBPATH = 'ya'
    YA_DEV_SUBPATH = 'devtools/ya-dev/ya'
    YA_OPTIONS_TPL = ('-v kiwi build '
                      '--dump-ctx=%(context_file)s '
                      '-S %(source_dir)s '
                      '-B %(build_dir)s '
                      '-I %(symlinks_dir)s '
                      '--build=%(build_type)s '
                      '-j%(threads)s'
                      )

    PROTOS_SVN_PATH = 'yweb/robot/kiwi/protos'
    PROTO_SFX = '.proto'
    PROTO_META_NAME = 'tuplemeta'
    PROTO_META_FILE = PROTO_META_NAME + PROTO_SFX
    PROTO_MQ_NAME = 'metaquery'
    PROTO_MQ_FILE = PROTO_MQ_NAME + PROTO_SFX
    PROTOC_OUT_DIR = 'pb_out'
    PB_PY_SFX = '_pb2'
    PB_META_MODULE = PROTO_META_NAME + PB_PY_SFX
    PB_MQ_MODULE = PROTO_MQ_NAME + PB_PY_SFX
    CMAKE_LISTS_FILE = 'CMakeLists.txt'
    TRIGGERS_DIR = 'triggers'
    TRIGGERS_OLD_VERSION = 1
    TRIGGERS_BRANCH_ENV_VAR = 'TRIGGER_BRANCH'
    EXPAND_SCRIPT = 'expand-versions.sh'
    EXPAND_LOG_PREFIX = 'expand'

    release_to = ['zagevalo@yandex-team.ru']
    archs = ['freebsd', 'linux']
    modes = ['lib_mode', 'data_mode']

    input_parameters = form.input_parameters

    def on_enqueue(self):
        """
            Create necessary resource(s) and save them in context before
            executing the task.
            @return: None
        """
        channel.task = self
        if self.ctx.get('multi_arch'):
            parent_resources = {}
            for arch in self.archs:
                parent_resources[arch] = self.mk_resource(arch=arch)
            self.ctx['parent_resources'] = parent_resources
        else:
            self.ctx['pre_defined_resource'] = self.mk_resource()

        # FIXME(rdna@): Sandbox can't handle correctly form with complicated
        #               dependencies so we should help with it.
        if self.ctx.get('task_mode') == 'use_ya_impl':
            for ctx_field in ['triggers_specific_str', 'build_set', ]:
                if self.ctx.get(ctx_field):
                    del self.ctx[ctx_field]

    def mk_resource(self, arch=None):
        """
            Create resource with path and description set according to @arch.
            @return: resource id
        """
        if arch:
            resource_path = os.path.join(self.RESULT_FOLDER, arch)
        else:
            arch = self.arch
            resource_path = self.RESULT_FOLDER
        resource = self.create_resource(
            description='%s: %s' % (resource_path, self.descr),
            resource_path=resource_path,
            resource_type=resource_types.KWTRIGGER_FOLDER,
            arch=arch
        )
        return resource.id

    def prepare_ctx(self):
        """
            Some context variables have to be additionally handled before
            using. Do it here.
        """
        self.ctx['build_triggers_arcadia_url'] = util.normalize_svn_url(
            self.ctx.get('build_triggers_arcadia_url', '')
        )
        self.ctx['triggers_svn_meta_path'] = self.ctx.get('triggers_svn_meta_path', '').strip().rstrip('/')
        self.ctx['triggers_svn_path'] = self.ctx.get('triggers_svn_path', '').strip().rstrip('/')

    def set_triggers_info(self):
        # TODO: Document the method
        targets_list = []
        triggers_list = []
        build_set = self.ctx['build_set']
        triggers_svn_path = self.ctx['triggers_svn_path']
        triggers_src_dir = util.get_arc_path(
            self.ctx['build_triggers_arcadia_url'], triggers_svn_path
        )
        self.set_triggers_tree_info(triggers_src_dir)
        triggers_name2udf = self.ctx['triggers_name2udf']
        triggers_targets = self.ctx['triggers_targets']
        if build_set == 'meta':
            triggers_svn_meta_path = self.ctx['triggers_svn_meta_path']
            triggers_meta_dir = util.get_arc_path(
                self.ctx['build_triggers_arcadia_url'],
                triggers_svn_meta_path
            )
            targets_list = [triggers_svn_meta_path]
            triggers_meta_list = self.get_triggers_meta_list(triggers_meta_dir, triggers_svn_path)
            for trigger_name in triggers_meta_list:
                if not triggers_name2udf.get(trigger_name):
                    continue
                triggers_list.append(trigger_name)
        elif build_set == 'specific':
            triggers_specific_str = self.ctx['triggers_specific_str']
            if not triggers_specific_str:
                targets_list = [triggers_svn_path]
                triggers_list = triggers_name2udf.keys()
            else:
                triggers_specific_list = triggers_specific_str.split()
                for trigger_name in triggers_specific_list:
                    if not triggers_name2udf.get(trigger_name):
                        self.report_nonexistent_trigger(trigger_name)
                        continue
                    targets_list.append('%s/%s' % (triggers_svn_path, triggers_targets[trigger_name]))
                    triggers_list.append(trigger_name)
        else:
            raise TaskFail('Unknown build set: %s.' % build_set)
        self.set_triggers_branch_props()
        self.prepare_expand_ctx()
        self.prepare_pb_modules()
        resource_path = self.prepare_resource()
        if self.ctx.get('use_pregenerated_mq'):
            self.handle_mq_files(triggers_list, resource_path)
        else:
            self.handle_meta_files(triggers_list, resource_path)
        self.ctx['targets_list'] = targets_list
        self.ctx['triggers_list'] = triggers_list
        logging.info('Targets to build: %s' % targets_list)
        logging.info('Triggers to build: %s' % triggers_list)

    def on_execute(self):
        """
            If task is in multi_arch mode, create and manage subtasks to build
            triggers on different architectures. Otherwise do real work right
            here.
            @return: None
        """
        self.prepare_ctx()
        if self.ctx.get('multi_arch'):
            # We're building the task for more than one architecture so let's
            # create and manage subtasks for real work.
            subtasks = self.list_subtasks(load=True)
            if not subtasks:
                # We're about to create subtasks and go to sleep after that.
                subtasks = []
                for arch in self.archs:
                    logging.info('Create subtask for %s.' % arch)
                    sub_ctx = copy.deepcopy(self.ctx)
                    # We don't need notifications for subtasks.
                    sub_ctx['notify_via'] = ''
                    sub_ctx['notify_if_finished'] = ''
                    sub_ctx['notify_if_failed'] = ''
                    # Prevent infinite recursion in subtask.
                    sub_ctx['multi_arch'] = False
                    sub_ctx['parent_resource'] = self.ctx['parent_resources'][arch]
                    subtask = self.create_subtask(
                        task_type=self.type,
                        arch=arch, input_parameters=sub_ctx,
                        description='Subtask for task #%s on %s' % (self.id, arch)
                    )
                    subtasks.append(subtask.id)
                self.wait_all_tasks_stop_executing(subtasks)
            else:
                # We're waked up after subtasks have finished.
                build_subtasks = [subtask for subtask in subtasks if subtask.type == self.type]
                for subtask in build_subtasks:
                    logging.info('Check subtask %s.' % subtask.id)
                    if subtask.status == self.Status.FAILURE or subtask.status in (
                        self.Status.EXCEPTION, self.Status.TIMEOUT, self.Status.NO_RES
                    ):
                        raise TaskFail('Subtask %s failed.' % subtask.id)
                    for resource in self.list_resources():
                        logging.info('Check resource %s.' % resource.id)
                        if resource.type != resource_types.TASK_LOGS and \
                                not resource.is_ready():
                            raise TaskFail('Resource %s (%s) is not ready.' % (resource.id, resource.type))
        else:
            # We're building the task for one architecture, just do it.
            self.do_execute()

    def do_execute(self):
        self.do_build()

    def do_build(self):
        if self.ctx.get('task_mode') == 'use_ya_impl':
            self.do_build_ya_impl()
        elif self.ctx.get('task_mode') == 'use_sandbox_impl':
            self.do_build_sandbox_impl()
        else:
            raise TaskFail('Unknown task mode: %s.' % self.ctx.get('task_mode'))
        resource = channel.sandbox.get_resource(self.ctx['pre_defined_resource'])
        if self.ctx.get('parent_resource'):
            self.save_parent_task_resource(resource.path, self.ctx['parent_resource'])
        else:
            self.mark_resource_ready(resource)
        return resource

    def do_build_sandbox_impl(self):
        self.set_triggers_info()
        self.do_in_meta_mode()
        if self.ctx['build_set'] == 'meta':
            self.mk_deploy_files()

    def do_build_ya_impl(self):
        logging.info('Use "%s" to build everything.' % self.YA_SUBPATH)

        if not self.ctx.get('targets_str'):
            raise TaskFail('Build list if empty! Please set targets to build.')
        targets_list = self.ctx['targets_str'].split()
        logging.info('Target list: %s' % targets_list)

        build_svn_url = self.ctx['build_triggers_arcadia_url']
        self.ctx[consts.ARCADIA_URL_KEY] = build_svn_url
        arcadia_src_dir = self.get_arcadia_src_dir()
        self.ctx['arcadia_src_dir'] = arcadia_src_dir

        targets_dirs = [os.path.join(arcadia_src_dir, target)
                        for target in targets_list]
        targets_opts = ['-C %s' % target_dir for target_dir in targets_dirs]

        build_type = self.ctx['build_type']

        context_file = self.abs_path(self.CONTEXT_FILE)
        build_dir = self.abs_path(self.BUILD_DIR)
        symlinks_dir = self.abs_path(self.SYMLINKS_DIR)
        make_folder(build_dir)
        make_folder(symlinks_dir)

        ya_path = os.path.join(
            arcadia_src_dir,
            self.YA_SUBPATH if not self.ctx['use_dev_version'] else self.YA_DEV_SUBPATH
        )
        ya_options = self.YA_OPTIONS_TPL % {
            'context_file': context_file,
            'source_dir': arcadia_src_dir,
            'build_dir': build_dir,
            'symlinks_dir': symlinks_dir,
            'build_type': build_type,
            'threads': int(self.client_info['ncpu']),
        }

        self.set_triggers_branch_props()
        if self.ctx['triggers_props'].get('AllowedExecCrashCount'):
            ya_options += ' --allowed-exec-crash-count'

        ya_cmd = '%s %s %s' % (ya_path, ya_options, ' '.join(targets_opts))
        run_process(ya_cmd, log_prefix='ya_kiwi_build')

        with open(context_file) as context_fd:
            result_dir = json.load(context_fd)['result_dir']
        resource_id = self.ctx['pre_defined_resource']
        resource = channel.sandbox.get_resource(resource_id)
        logging.info('Move result from %s to resource directory.' % result_dir)
        shutil.move(result_dir, resource.path)
        self.mk_info_file()
        logging.info('Build finished.')

    def do_in_meta_mode(self):
        # TODO: Document the method

        logging.info('Create threads.')
        thread_cbs = {}     # ``cb'' is Control Block.
        self_ctx_lock = threading.Lock()
        arcadia_src_cv = threading.Condition(self_ctx_lock)
        for mode in self.modes:
            logging.info('Create thread for %s mode.' % mode)
            thread_cb = {'thread': None, 'result': None, }
            with self_ctx_lock:
                thread_ctx = copy.deepcopy(self.ctx)
            if mode == 'lib_mode':
                args = (thread_cb, self.do_in_lib_mode, thread_ctx)
                kwargs = {'arcadia_src_cv': arcadia_src_cv, 'self_ctx_lock': self_ctx_lock}
            elif mode == 'data_mode':
                args = (thread_cb, self.do_in_data_mode, thread_ctx,
                        arcadia_src_cv)
                kwargs = {}
            else:
                raise TaskFail('Function for mode %s is not defined.' % mode)

            thread = threading.Thread(
                target=util.runner,
                name=mode,
                args=args,
                kwargs=kwargs
            )
            thread.start()
            thread_cb['thread'] = thread
            thread_cbs[mode] = thread_cb
        logging.info('Threads are created.')

        util.wait_for_threads(thread_cbs)

    def do_in_lib_mode(self, ctx, arcadia_src_cv=None, self_ctx_lock=None):
        # TODO: Document the method
        if not ctx.get('pre_defined_resource'):
            raise TaskFail('There is no predefined resource.')
        resource_id = ctx['pre_defined_resource']
        resource = channel.sandbox.get_resource(resource_id)

        build_svn_url = ctx['build_triggers_arcadia_url']
        if not ctx.get('targets_list'):
            raise TaskFail('Build list if empty! Select triggers to build.')
        targets_list = ctx['targets_list']
        logging.info('KiWi-triggers target list: %s' % targets_list)

        # ``checkout_arcadia_from_url'' should be set in "global" context.
        # It is used in get_arcadia_src_dir() method.
        if self_ctx_lock:
            logging.info('Use lock to change global context.')
            with self_ctx_lock:
                self.ctx[consts.ARCADIA_URL_KEY] = build_svn_url
        else:
            self.ctx[consts.ARCADIA_URL_KEY] = build_svn_url

        source_revision = get_svn_info(build_svn_url)['entry_revision']
        logging.info('Source revision: %s' % source_revision)

        arcadia_src_dir = self.get_arcadia_src_dir()
        if arcadia_src_cv:
            logging.info('Add arcadia_src_dir to context.')
            arcadia_src_cv.acquire()
            self.ctx['arcadia_src_dir'] = arcadia_src_dir
            arcadia_src_cv.notify()
            arcadia_src_cv.release()
        else:
            self.ctx['arcadia_src_dir'] = arcadia_src_dir

        build_dir = self.abs_path(self.BUILD_DIR)
        make_folder(build_dir)
        os.chdir(build_dir)

        cmake_cmd = [
            'cmake', arcadia_src_dir,
            '-DCMAKE_BUILD_TYPE=%s' % ctx['build_type'],
            '-DMAKE_ONLY=%s' % ';'.join(targets_list)
        ]
        self._subprocess(cmake_cmd, wait=True, log_prefix='cmake_robot')

        for target in targets_list:
            target_dir = self.abs_path(os.path.join(build_dir, target))
            make_cmd = get_make_name(jobs_by_cores=True) + [
                '-C', target_dir, 'VERBOSE=1'
            ]
            self._subprocess(
                make_cmd, wait=True, log_prefix='make_robot_%s' % target.replace('/', '_')
            )

        self.cp_libs_to_resource(ctx, build_dir, resource)
        self.mk_info_file()
        self.set_info('Libraries are built successfully.')

    def do_in_data_mode(self, ctx, arcadia_src_cv=None):
        """
            Copy data files for triggers.
            @return: None
        """
        if not ctx.get('pre_defined_resource'):
            raise TaskFail('There is no predefined resource.')
        resource_id = ctx['pre_defined_resource']
        resource = channel.sandbox.get_resource(resource_id)

        pb_mq_module = ctx['pb_mq_module']
        triggers_list = ctx['triggers_list']
        triggers_name2udf = ctx['triggers_name2udf']

        data_cache = DataCache()
        logging.info('Copy data for triggers to %s.', resource.path)
        for trigger_name in TriggersData.names():
            if not (trigger_name in triggers_list):
                logging.info('Copy[%s]: skip (not selected).', trigger_name)
                continue
            trigger_udf = triggers_name2udf[trigger_name]
            trigger_dir = os.path.join(resource.path, trigger_udf)
            trigger_datapath = TriggersData.get_datapath(
                trigger_dir, trigger_name
            )
            trigger_mq_file = os.path.join(trigger_dir, cfg.MQ_META_FILE)
            mq_msg = pbmsg.get_mq(pb_mq_module, trigger_mq_file)
            for file_item in TriggersData.get_files(trigger_name):
                src = file_item['src']
                dst = os.path.join(
                    trigger_datapath, file_item.get('dst', os.path.basename(src))
                )
                post_hook_name = file_item.get('post_hook_name')
                make_folder(os.path.dirname(dst))
                if TriggersData.is_shared(trigger_name) and os.path.exists(dst):
                    logging.info('Copy[%s]: skip shared %s' % (trigger_name, dst))
                    continue
                if data_cache.is_cached(src):
                    data_cache.load(src, dst)
                    continue
                logging.info('Copy[%s]: %s -> %s' % (trigger_name, src, dst))
                scheme_cp = SchemeCp(src, dst, task=self)
                if scheme_cp.need_arc_info and not SchemeCp.is_arc_info_set:
                    SchemeCp.set_arc_info(
                        arc_url=ctx['build_triggers_arcadia_url'],
                        arc_src_dir=ctx.get('arcadia_src_dir'),
                        arc_src_cv=arcadia_src_cv
                    )
                scheme_cp.run()
                tmp = scheme_cp.get_tmp()
                if post_hook_name:
                    post_hook = PostCpHook(post_hook_name, dst)
                    post_hook.run()
                    if post_hook.is_cachable():
                        data_cache.save(src, dst, tmp)
                else:
                    data_cache.save(src, dst, tmp)
            mq_msg = metaquery.add_files_to_udf(trigger_name, trigger_dir, mq_msg)
            pbmsg.save_common_msg(mq_msg, trigger_mq_file)
        logging.info('Data files are copied successfully.')
        self.set_info('Data files are copied successfully.')

    def cp_libs_to_resource(self, ctx, build_dir, resource):
        """
            Finds trigger libraries in @build_dir and copies them into
            @resource.
            :return: nothing.
        """
        build_lib_dir = os.path.join(build_dir, 'lib')
        if not os.path.exists(build_lib_dir):
            raise TaskFail('There is no %s directory' % build_lib_dir)

        logging.info('Copy trigger libraries from %s to %s' % (build_lib_dir, resource.path))

        lib_re = re.compile(r'.*/(?P<filename>lib(?P<udf>[^/]+).so)$')
        for trigger_path in glob.glob(os.path.join(build_lib_dir, 'lib*.so')):
            search_result = lib_re.search(trigger_path)
            if not search_result:
                raise TaskFail('Unexpected name of trigger: %s' % trigger_path)
            trigger_filename = search_result.group('filename')
            trigger_udf = search_result.group('udf')
            if not self.ctx['triggers_udf2name'].get(trigger_udf):
                logging.info('Skip non-UDF library %s.' % trigger_filename)
                continue
            trigger_dir = os.path.join(resource.path, trigger_udf)
            trigger_realpath = os.path.realpath(trigger_path)
            trigger_dstpath = os.path.join(trigger_dir, trigger_filename)
            if not os.path.exists(trigger_dstpath):
                os.link(trigger_realpath, trigger_dstpath)

    def mk_info_file(self):
        """
            Creates info file in pre-defined resource to save current task id,
            branch name and revision.
            :return: nothing.
        """
        svn_url = self.ctx['build_triggers_arcadia_url']
        branch_name = util.get_branch_name(svn_url)
        revision = util.get_revision(svn_url)

        resource_id = self.ctx['pre_defined_resource']
        resource = channel.sandbox.get_resource(resource_id)

        info_text = '%s\n%s\n%s\n' % (self.id, branch_name, revision)
        info_file = os.path.join(resource.path, cfg.TASKINFO_FILE_NAME)

        logging.info('Create info file %s:\n%s.' % (info_file, info_text))
        with open(info_file, 'w') as info_fd:
            info_fd.write(str(info_text))

    def mk_deploy_files(self):
        """
           Makes deployment metaquery-files for registering and enabling stored
           procedures, releasing procedures and exportes.
           :return: nothing.
        """
        logging.info('Make deploy files.')

        resource_id = self.ctx['pre_defined_resource']
        arcadia_src_dir = self.ctx['arcadia_src_dir']
        triggers_svn_meta_path = self.ctx['triggers_svn_meta_path']
        use_pregenerated_mq = self.ctx['use_pregenerated_mq']
        pb_mq_module = self.ctx['pb_mq_module']
        triggers_udf_names = self.ctx['triggers_udf_names']
        triggers_props = self.ctx['triggers_props']

        resource = channel.sandbox.get_resource(resource_id)
        deploy_dir = os.path.join(resource.path, cfg.DEPLOY_DIR)
        make_folder(deploy_dir)

        if use_pregenerated_mq:
            queries = deploy.Queries(
                arcadia_src_dir, triggers_svn_meta_path, pb_mq_module
            )
        else:
            queries = deploy.Queries(arcadia_src_dir, triggers_svn_meta_path)

        queries.load()
        queries.install_prog_files(deploy_dir, self.expand_versions)
        queries.install_mq_files(deploy_dir, triggers_props)

        mq_maker = deploy.MetaQueryMaker(use_pregenerated_mq, deploy_dir)
        mq_maker.make(pb_mq_module, triggers_udf_names, queries, triggers_props)

        logging.info('MetaQuery files are prepared.')
        self.set_info('MetaQuery files are prepared.')

    def report_nonexistent_trigger(self, trigger_name):
        # TODO: Document the method
        logging.warning(
            'Trigger %s does not exist in %s.' % (
                trigger_name, self.ctx['build_triggers_arcadia_url']
            )
        )
        self.set_info(
            'WARNING!!! %s trigger is chosen to built but not found in %s!' % (
                trigger_name, self.ctx['triggers_svn_path']
            )
        )

    def prepare_expand_ctx(self):
        """
            Add to context all the things necessary for expanding query
            scripts: expand script, version to substitute, etc.
            @return: None.
        """
        if not self.ctx.get('expand_env'):
            triggers_branch = self.ctx['triggers_branch']
            env = os.environ.copy()
            env['LD_LIBRARY_PATH'] = ''     # like in run_process() by default
            env[self.TRIGGERS_BRANCH_ENV_VAR] = triggers_branch
            logging.info('Add %s = %s in expand environment.' % (self.TRIGGERS_BRANCH_ENV_VAR, triggers_branch))
            self.ctx['expand_env'] = env
        if not self.ctx.get('expand_script'):
            logging.info('Add expand script in context.')
            self.ctx['expand_script'] = util.get_arc_path(
                self.ctx['build_triggers_arcadia_url'],
                os.path.join(cfg.QUERIES_SVN_PATH, self.EXPAND_SCRIPT),
                isfile=True
            )

    def expand_versions(self, in_file, out_file):
        """
            Expand versions in @in_file and save result to @out_file.
            @return: @out_file.
        """
        logging.info('Expand %s file.' % in_file)
        expand_env = self.ctx['expand_env']
        expand_script = self.ctx['expand_script']
        expand_cmd = '%s < %s -n > %s' % (expand_script, in_file, out_file)
        run_process(
            expand_cmd, log_prefix=self.EXPAND_LOG_PREFIX, shell=True,
            environment=expand_env
        )
        return out_file

    def set_triggers_branch_props(self):
        """
            Sets branch-specific info for triggers in context: triggers_branch,
            triggers_props that depends on triggers branch group, etc.
            @return: None
        """
        svn_url = self.ctx['build_triggers_arcadia_url']

        triggers_branch_group = util.get_branch_group(svn_url)
        triggers_props = branch_cfg.group_props.get(triggers_branch_group, {})

        triggers_branch = util.get_normalized_branch_name(svn_url)
        revision = util.get_revision(svn_url)
        triggers_props['tag'] = '%s.r%s' % (triggers_branch, revision)

        self.ctx['triggers_branch'] = triggers_branch
        self.ctx['triggers_props'] = triggers_props
        logging.info('Set triggers branch in context: %s.' % triggers_branch)
        logging.info('Set triggers properties in context: %s.' % triggers_props)

    def get_proto_set(self, arc_root, proto_file, result=set()):
        """
            Make set of all the proto-files required to compile @proto_file in
            @arc_root, including @proto_file itself, and merge the set with
            @result set.
            @return: @result set after merging.
        """
        logging.info('Get dependencies from proto file %s.' % proto_file)
        result.add(proto_file)
        proto_path = os.path.join(arc_root, proto_file)
        if not os.path.isfile(proto_path):
            proto_path = util.get_arc_path(
                self.ctx['build_triggers_arcadia_url'], proto_file,
                isfile=True
            )
        with open(proto_path) as proto_fd:
            proto_data = proto_fd.read()
        import_re = re.compile(r'^\s*import\s*"(?P<file>\S+)"\s*;\s*$', re.M)
        for search_res in import_re.finditer(proto_data):
            import_file = search_res.group('file')
            logging.info('Proto import: %s -> %s' % (proto_file, import_file))
            self.get_proto_set(arc_root, import_file, result)
        return result

    def prepare_pb_modules(self):
        # TODO: Document the method
        protos_src_dir = util.get_arc_path(
            self.ctx['build_triggers_arcadia_url'], self.PROTOS_SVN_PATH
        )
        arc_root = protos_src_dir.replace('/' + self.PROTOS_SVN_PATH, '')
        proto_meta_file = os.path.join(self.PROTOS_SVN_PATH, self.PROTO_META_FILE)
        proto_mq_file = os.path.join(self.PROTOS_SVN_PATH, self.PROTO_MQ_FILE)
        proto_files_set = self.get_proto_set(arc_root, proto_meta_file)
        proto_files_set = self.get_proto_set(arc_root, proto_mq_file, proto_files_set)
        proto_paths_list = [
            os.path.join(arc_root, proto_file)
            for proto_file in proto_files_set
        ]
        proto_paths = ' '.join(proto_paths_list)
        # TODO: Make sure that protoc version from sandbox is compatible with
        #       python protobuf API from skynet python.
        protoc_resources = apihelpers.get_resources_with_attribute(
            arch=self.arch,
            resource_type=resource_types.PROTOC_EXECUTABLE, limit=1,
            attribute_name='linked_statically', attribute_value='yes'
        )
        if not protoc_resources:
            raise TaskFail('PROTOC_EXECUTABLE resource for %s is not found.' % self.arch)
        self.sync_resource(protoc_resources[0].id)
        protoc_local_resource = channel.sandbox.get_resource(protoc_resources[0].id)
        protoc_path = protoc_local_resource.path
        protoc_import_path = self.get_task_abs_path(self.id)
        protoc_out_dir = self.abs_path(self.PROTOC_OUT_DIR)
        make_folder(protoc_out_dir)

        protoc_cmd = '%s -I%s --python_out=%s %s' % (
            protoc_path, protoc_import_path, protoc_out_dir, proto_paths
        )
        run_process(protoc_cmd, log_prefix=os.path.basename(protoc_path))

        # To treat generated by protoc directory structure as a package we have
        # to have __init__.py in every subdirectory.
        for dirpath, _, _ in os.walk(protoc_out_dir, onerror=True):
            with open(os.path.join(dirpath, '__init__.py'), 'w'):
                pass

        sys.path.append(protoc_out_dir)
        pb_module_prefix = self.PROTOS_SVN_PATH.replace('/', '.')
        pb_meta_module = '%s.%s' % (pb_module_prefix, self.PB_META_MODULE)
        pb_mq_module = '%s.%s' % (pb_module_prefix, self.PB_MQ_MODULE)
        self.ctx['pb_meta_module'] = pb_meta_module
        self.ctx['pb_mq_module'] = pb_mq_module
        logging.info('PB meta module: %s.' % pb_meta_module)
        logging.info('PB mq module: %s.' % pb_mq_module)

    def prepare_resource(self):
        """
            Creates resource directory.
            :return: created directory.
        """
        resource_id = self.ctx['pre_defined_resource']
        resource = channel.sandbox.get_resource(resource_id)
        logging.info('Create resource directory for KiWi triggers.')
        make_folder(resource.path)
        return resource.path

    def handle_mq_files(self, triggers_list, resource_path):
        """Gets all mq template files for every trigger from @triggers_list,
        generate real mq files from them and saves the result in @resource_path.
        :return: nothing.
        """
        triggers_props = self.ctx['triggers_props']
        triggers_src_dirs = self.ctx['triggers_src_dirs']
        triggers_name2udf = self.ctx['triggers_name2udf']
        triggers_udf_names = set()

        pb_mq_module = self.ctx['pb_mq_module']

        for trigger_name, trigger_src_dir in triggers_src_dirs.items():
            if trigger_name not in triggers_list:
                logging.info('Skip %s (not chosen to be built).' % trigger_name)
                continue

            trigger_udf = triggers_name2udf[trigger_name]
            trigger_dir = os.path.join(resource_path, trigger_udf)
            make_folder(trigger_dir)

            logging.info('Handle %s for %s.' % (cfg.MQ_META_FILE, trigger_name))
            trigger_mq_meta_tpl = os.path.join(trigger_src_dir, cfg.MQ_META_TPL)
            util.check_trigger_file(trigger_name, trigger_mq_meta_tpl)
            mq_meta_msg = pbmsg.get_mq(pb_mq_module, trigger_mq_meta_tpl)
            mq_meta_msg = metaquery.handle_udf_statements(triggers_props['tag'], mq_meta_msg)
            trigger_mq_meta_file = os.path.join(trigger_dir, cfg.MQ_META_FILE)
            pbmsg.save_common_msg(mq_meta_msg, trigger_mq_meta_file)

            trigger_udf_names = metaquery.get_trigger_udf_names(mq_meta_msg)
            triggers_udf_names.update(trigger_udf_names)

            logging.info('Make %s for %s.' % (cfg.MQ_ENABLE_FILE, trigger_name))
            mq_enable_msg = pbmsg.get_mq(pb_mq_module)
            for trigger_udf_name in trigger_udf_names:
                mq_enable_msg = metaquery.enable_trigger(
                    trigger_udf_name, triggers_props['tag'], mq_enable_msg
                )
            trigger_mq_enable_file = os.path.join(trigger_dir, cfg.MQ_ENABLE_FILE)
            pbmsg.save_common_msg(mq_enable_msg, trigger_mq_enable_file)

        self.ctx['triggers_udf_names'] = triggers_udf_names
        logging.info('Triggers UDF names: %s.' % triggers_udf_names)

    def handle_meta_files(self, triggers_list, resource_path):
        # TODO: Document the method
        triggers_props = self.ctx['triggers_props']
        triggers_src_dirs = self.ctx['triggers_src_dirs']
        triggers_name2udf = self.ctx['triggers_name2udf']
        triggers_udf2name = self.ctx['triggers_udf2name']
        triggers_udf_names = set()

        pb_meta_module = self.ctx['pb_meta_module']
        pb_mq_module = self.ctx['pb_mq_module']
        mq_module = __import__(pb_mq_module, fromlist=['TMetaQuery'])

        is_parsed = False
        for trigger_name, trigger_src_dir in triggers_src_dirs.items():
            if trigger_name not in triggers_list:
                logging.info('Skip %s (not chosen to be built).' % trigger_name)
                continue

            trigger_svn_meta_file = os.path.join(trigger_src_dir, cfg.PB_META_FILE)
            util.check_trigger_file(trigger_name, trigger_svn_meta_file)

            meta_msg = pbmsg.get_meta(pb_meta_module, trigger_svn_meta_file)
            mq_msg = mq_module.TMetaQuery()

            meta_msg_Udfs_count = len(meta_msg.Udfs)
            if meta_msg_Udfs_count < 1:
                raise TaskFail(
                    '%s contains %d Udfs section(s). There have to be at least one.' % (
                        trigger_svn_meta_file, meta_msg_Udfs_count
                    )
                )

            trigger_udf_names = set()

            for udfs_i, udfs_msg in enumerate(meta_msg.Udfs):
                trigger_qsname = udfs_msg.Name
                trigger_udf_names.add(trigger_qsname)
                logging.info('Parse UDF %s.' % trigger_qsname)

                udfs_msg_Versions_count = len(udfs_msg.Versions)
                if udfs_msg_Versions_count != 1:
                    raise TaskFail(
                        '%s contains %d Versions section(s). There should be exactly one section.' % (
                            trigger_svn_meta_file, udfs_msg_Versions_count
                        )
                    )

                versions_msg = udfs_msg.Versions[0]
                versions_msg.Version = self.TRIGGERS_OLD_VERSION

                search_res = re.search(
                    r'/*(?P<trigger_lib>lib(?P<trigger_udf>[^/]+).so)',
                    versions_msg.SoName
                )
                if search_res:
                    trigger_lib = search_res.group('trigger_lib')
                    trigger_udf = search_res.group('trigger_udf')
                else:
                    raise TaskFail(
                        'Unexpected SoName for %s: %s.' % (trigger_name, versions_msg.SoName)
                    )

                if trigger_name != triggers_udf2name.get(trigger_udf):
                    raise TaskFail(
                        'Library name for %s in %s and %s seems to be different.' % (
                            trigger_name, self.CMAKE_LISTS_FILE, cfg.PB_META_FILE
                        )
                    )

                trigger_dir = os.path.join(self.TRIGGERS_DIR, trigger_udf)
                versions_msg.SoName = os.path.join(trigger_dir, trigger_lib)
                versions_msg.DataPath = TriggersData.get_datapath_str(
                    trigger_dir, trigger_name, trigger_qsname,
                    versions_msg.DataPath
                )
                if versions_msg.BaseDir:
                    versions_msg.BaseDir = TriggersData.get_basedir(
                        trigger_dir, trigger_name, trigger_qsname
                    )

                for files_msg in versions_msg.Files:
                    files_msg.RelativePath = PathEval(files_msg.RelativePath).get_path()

                mq_msg = metaquery.add_udf(triggers_props, udfs_msg, mq_msg)

            triggers_udf_names.update(trigger_udf_names)
            trigger_udf = triggers_name2udf[trigger_name]
            # FIXME(rdna@): Resolve lock mask in meta_msg.
            pbmsg.save_all(resource_path, trigger_udf, meta_msg, mq_msg)
            is_parsed = True

        if not is_parsed:
            raise TaskFail('Cannot parse meta files.')

        self.ctx['triggers_udf_names'] = triggers_udf_names
        logging.info('Triggers UDF names: %s.' % triggers_udf_names)

    def set_triggers_tree_info(self, triggers_src_dir):
        """
            Walks through @triggers_src_dir and inits the following dictionaries
            in the current context:
            * direct and reverse maps for ``trigger svn directory name''
              (trigger_name) and ``UDF name from cmake-file'' (trigger_udf)
              pairs;
            * target directories starting from @triggers_src_dir
              (triggers_targets);
            * triggers source directories starting from / (triggers_src_dir).
            :return: None.
        """
        triggers_name2udf = {}
        triggers_udf2name = {}
        triggers_targets = {}
        triggers_src_dirs = {}
        for dirpath, _, filenames in os.walk(triggers_src_dir, onerror=True):
            trigger_cmake_file = None
            for filename in filenames:
                if filename == self.CMAKE_LISTS_FILE:
                    trigger_cmake_file = os.path.join(dirpath, filename)
                    break
            if not trigger_cmake_file:
                continue
            trigger_name = os.path.basename(dirpath)
            trigger_target = dirpath[len(triggers_src_dir) + 1:]
            trigger_cmake_fd = open(trigger_cmake_file)
            for line in trigger_cmake_fd.readlines():
                search_res = re.search(r'(DLL|LIBRARY|UDF)\(([^\)]+)\)', line)
                if search_res:
                    trigger_udf = search_res.group(2)
                    triggers_name2udf[trigger_name] = trigger_udf
                    triggers_udf2name[trigger_udf] = trigger_name
                    triggers_targets[trigger_name] = trigger_target
                    # Every trigger should have meta file. Here we just make a
                    # list of meta files which should exist, real existence
                    # will be checked later.
                    triggers_src_dirs[trigger_name] = dirpath
                    break
            trigger_cmake_fd.close()
        if len(triggers_name2udf) == 0:
            raise TaskFail('Cannot set triggers_name2udf map.')
        if len(triggers_udf2name) == 0:
            raise TaskFail('Cannot set triggers_udf2name map.')
        self.ctx['triggers_name2udf'] = triggers_name2udf
        self.ctx['triggers_udf2name'] = triggers_udf2name
        self.ctx['triggers_targets'] = triggers_targets
        self.ctx['triggers_src_dirs'] = triggers_src_dirs
        logging.info('Map->name2udf: %s' % triggers_name2udf)
        logging.info('Map->udf2name: %s' % triggers_udf2name)
        logging.info('All targets: %s' % triggers_targets)
        logging.info('Triggers source directories: %s' % triggers_src_dirs)

    def get_triggers_meta_list(self, meta_dir, triggers_svn_path):
        # TODO: Document the method
        meta_file = os.path.join(meta_dir, self.CMAKE_LISTS_FILE)
        with open(meta_file) as meta_fd:
            meta_out = meta_fd.read()
        trigger_name_re = re.compile(
            r'^\s*' + triggers_svn_path + r'.*/(?P<trigger_name>\S+)\s*\n', flags=re.M
        )
        triggers_meta_list = []
        for search_res in trigger_name_re.finditer(meta_out):
            triggers_meta_list.append(search_res.group('trigger_name'))
        if not triggers_meta_list:
            raise TaskFail('Cannot get list of triggers from %s.' % meta_file)
        logging.info('Triggers meta list: %s.' % triggers_meta_list)
        return triggers_meta_list

    def arcadia_info(self):
        svn_url = self.ctx.get('build_triggers_arcadia_url', '')
        search_result = re.search(r'arc/(.*)/arcadia', svn_url)
        if search_result:
            arcadia_branch = search_result.group(1)
        else:
            arcadia_branch = None
        return None, arcadia_branch, arcadia_branch


__Task__ = BuildKiwiTriggers

# vim:set ts=4 sw=4 et:
