"""Interface to work with data cache.
"""

import logging
import os

from sandbox.projects.BuildKiwiTriggers import util


class DataCache(object):
    """Class that implements data cache.
    """

    def __init__(self):
        logging.info('Initialize data cache.')
        self.store = {}

    def save(self, src, dst, tmp):
        """Saves @dst value to cache for key @src. If @tmp is set, all
        dir-entries from @tmp will be found and the cache will contain
        corresponding entries from @dst.
        :return: Nothing.
        """
        logging.info('Save %s to data cache.' % src)
        if tmp:
            self.store[src] = []
            for entry_name in os.listdir(tmp):
                entry_dst = os.path.join(dst, entry_name)
                self.store[src].append(entry_dst)
        else:
            self.store[src] = dst
        logging.info('Data cache for %s: %s.' % (src, self.store[src]))

    def is_cached(self, src):
        """Checks whether cache contains value for key @src or not.
        :return: True if key is found, False otherwise.
        """
        logging.info('Find %s in data cache.' % src)
        return self.store.get(src)

    def load(self, src, dst):
        """Loads cache by key @src to directory @dst.
        :return: Nothing.
        """
        logging.info('Load %s from cache: %s.' % (dst, src))
        if isinstance(self.store[src], list):
            for entry_src in self.store[src]:
                entry_name = os.path.basename(entry_src)
                entry_dst = os.path.join(dst, entry_name)
                util.fast_copy(entry_src, entry_dst, copy_function=util.symlink_rel)
        else:
            util.fast_copy(self.store[src], dst, copy_function=util.symlink_rel)
