"""Class for copying data files using different schemes.

Interface summary:

    from data_scheme import SchemeCp

    scheme_cp = SchemeCp(src, dst, task)    # create SchemeCp object
    scheme_cp.run()                         # run copy using specific scheme
"""

import logging
import os
import re

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail

from sandbox.projects.common import apihelpers

from sandbox.projects.BuildKiwiTriggers import util
from sandbox.projects.BuildKiwiTriggers.backward_compatibility_subs import bc_subs


RSYNC_TRIES = 3


Scheme = util.enum(
    RSYNC='rsync',
    SANDBOX='sandbox',                  # Sandbox resource
    SVN_BRANCH_ROOT='svn_branch_root',  # Branch root on svn server
    WC_FILE='wc_file',                  # ``WC'' - Working Copy
)


def sandbox_url(resource_type, resource_subpath=''):
    """Makes Sandbox resource URL from @resource_type and @resource_subpath.
    :return: Sandbox resource URL string.
    """
    return Scheme.SANDBOX + '://' + str(resource_type) + resource_subpath


class SchemeCp(object):
    """Class for copying data files using different schemes.
    """

    arc_url = None
    arc_src_dir = None
    arc_src_cv = None
    is_arc_info_set = False

    branch_subpath_re = None
    sandbox_resource_re = None
    scheme_re = None
    wc_file_re = None

    def __init__(self, src, dst, task=None):
        """Initialize members for current instance.
        """
        self.src = src
        self.dst = dst
        self.tmp = None  # For a scheme that can't copy @src directly to @dst.
        self.task = task
        self.need_arc_info = False
        self.set_scheme()

    def set_scheme(self):
        """Parse ``src'' member and set ``scheme'' member for current instance.
        """
        if not SchemeCp.scheme_re:
            SchemeCp.scheme_re = re.compile(r'^(?P<scheme>.+?)://')
        search_res = SchemeCp.scheme_re.search(self.src)
        if not search_res:
            raise TaskFail('Unexpected scheme for %s.' % self.src)
        self.scheme = search_res.group('scheme')
        logging.info('Set scheme %s for %s.' % (self.scheme, self.src))
        if self.scheme in (Scheme.SVN_BRANCH_ROOT, Scheme.WC_FILE):
            self.need_arc_info = True

    @staticmethod
    def set_arc_info(arc_url=None, arc_src_dir=None, arc_src_cv=None):
        """Set static members to keep info about arcadia.
        """
        SchemeCp.arc_url = arc_url
        SchemeCp.arc_src_dir = arc_src_dir
        SchemeCp.arc_src_cv = arc_src_cv
        SchemeCp.is_arc_info_set = True

    def run(self):
        """Run proper copying method according to ``scheme''.
        """
        if self.scheme == Scheme.RSYNC:
            self.cp_rsync()
        elif self.scheme == Scheme.SANDBOX:
            self.cp_sandbox()
        elif self.scheme == Scheme.WC_FILE:
            self.cp_work_copy()
        elif self.scheme == Scheme.SVN_BRANCH_ROOT:
            self.cp_svn()
        else:
            raise TaskFail('Scheme %s is not supported (src: %s).' % (self.scheme, self.src))

    def cp_rsync(self):
        """Copy from rsync server.
        """
        real_srcs = util.get_real_urls(self.src, num=RSYNC_TRIES)
        util.repeat_func(
            self.task.remote_copy,
            '<iterator>', real_srcs,
            src='<iterator>', dst=os.path.dirname(self.dst),
        )

    def cp_sandbox(self):
        """Copy from Sandbox resource.
        """
        if not SchemeCp.sandbox_resource_re:
            SchemeCp.sandbox_resource_re = re.compile(
                self.scheme +
                r'://(?P<resource_type>[^/]+)(/+(?P<subpath>.*))?$'
            )
        search_res = SchemeCp.sandbox_resource_re.search(self.src)
        if not search_res:
            raise TaskFail('Invalid sandbox path: %s.' % self.src)
        resource_type = search_res.group('resource_type')
        src_subpath = search_res.group('subpath')
        remote_resource = apihelpers.get_last_resource(resource_type)
        channel.task.sync_resource(remote_resource.id)
        local_resource = channel.sandbox.get_resource(remote_resource.id)
        # Trailing ``/'' in self.src is important.
        if src_subpath is None:
            local_src = local_resource.path
        else:
            local_src = os.path.join(local_resource.path, src_subpath)
            if local_src.endswith('/'):
                self.tmp = local_src
        util.fast_copy(local_src, self.dst)

    def cp_work_copy(self):
        """ Copy from local working copy of arcadia.
        """
        if not SchemeCp.wc_file_re:
            SchemeCp.wc_file_re = re.compile(self.scheme + r'://arcadia')
        if not SchemeCp.arc_src_dir:
            logging.info('Wait for arcadia_src_dir in context.')
            SchemeCp.arc_src_cv.acquire()
            while not self.task.ctx.get('arcadia_src_dir'):
                SchemeCp.arc_src_cv.wait()
            SchemeCp.arc_src_cv.release()
            logging.info('Obtain arcadia_src_dir from context .')
            SchemeCp.arc_src_dir = self.task.ctx['arcadia_src_dir']
        wc_src = SchemeCp.wc_file_re.sub(SchemeCp.arc_src_dir, self.src)
        if not os.path.exists(wc_src):
            wc_src = self.__try_old_path(wc_src)
        util.fast_copy(wc_src, self.dst)

    def cp_svn(self):
        """Copy from root of specific svn branch.
        """
        if not SchemeCp.branch_subpath_re:
            SchemeCp.branch_subpath_re = re.compile(
                Scheme.SVN_BRANCH_ROOT +
                r'://(?P<branch_subpath>.*)$'
            )
        search_res = SchemeCp.branch_subpath_re.search(self.src)
        if not search_res:
            raise TaskFail('Invalid path in branch: %s' % self.src)
        branch_subpath = search_res.group('branch_subpath')
        util.get_arc_path(
            SchemeCp.arc_url, branch_subpath, local_path=self.dst,
            use_branch_root=True, isfile=True
        )

    def get_tmp(self):
        """Gets temporary results of copying if any.
        :return: @tmp member of an instance.
        """
        return self.tmp

    def __try_old_path(self, path):
        """Tries to use old path to data file instead of @path to support both
        trunk and branches after moving kiwi_triggers/* to kiwi_queries/ in
        trunk.
        :return: old path if it exists, @path otherwise.
        """
        for new_str, old_str in bc_subs.iteritems():
            if path.find(new_str) != -1:
                old_path = path.replace(new_str, old_str)
                if os.path.exists(old_path):
                    logging.info('Use old path %s instead of %s.' % (old_path,
                                                                     path))
                    return old_path
                else:
                    return path
        return path
