"""Classes to generate deployment metaquery-files.
"""

import logging
import os
import re

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects.BuildKiwiTriggers import cfg
from sandbox.projects.BuildKiwiTriggers import metaquery
from sandbox.projects.BuildKiwiTriggers import pbmsg


class TxtFile(object):
    """Common interface to work with text files.
    """

    def __init__(self, path):
        self.path = path

    def check(self):
        """Checks whether file exists or not.
        :return: True if the file exists, False otherwise.
        """
        if not os.path.isfile(self.path):
            # Some meta directories may not have some deploy-file(s). It's OK.
            logging.info('File %s does not exist.' % self.path)
            return False
        return True

    def get_lines(self):
        """Gets all useful (non-blank and non-comment) lines from the file.
        :return: iterator yielding a line from the file.
        """
        comment_re = re.compile(r'\s*#.*$')
        with open(self.path) as fd:
            for line in fd.readlines():
                clean_line = comment_re.sub('', line).strip()
                if not clean_line:
                    # Line is empty or contains just comments.
                    continue
                yield clean_line


class LstFile(TxtFile):
    """Class to work with lst-files from building meta-directories.
    """

    def __init__(self, path, queries_dir):
        TxtFile.__init__(self, path)
        self.queries_dir = queries_dir
        self.query_files = set()

    def get_query_files(self):
        """Parses appropriate lst-file and make a set of query files from it.
        If there is a directory in lst-file, then all the files from it will be
        added to the set.
        :return: set with relative paths to query files.
        """
        logging.info('Parse lst-file %s.' % self.path)

        if not self.check():
            return self.query_files

        for line in self.get_lines():
            query_path = os.path.join(self.queries_dir, line)
            if os.path.isdir(query_path):
                for dirpath, _, filenames in os.walk(query_path, onerror=True):
                    for filename in filenames:
                        query_abs_path = os.path.join(dirpath, filename)
                        rel_pos = query_abs_path.find(line)
                        query_file = query_abs_path[rel_pos:]
                        self.query_files.add(query_file)
            else:
                query_file = line
                self.query_files.add(query_file)

        logging.info('Query files in %s: %s' % (self.path, self.query_files))

        return self.query_files


class ExportsAddFile(TxtFile):
    """Class to work with additional export-files from building
    meta-directories.
    """

    def __init__(self, path):
        TxtFile.__init__(self, path)
        self.exports_aliases = {}
        self.mq_files = set()

    def parse(self):
        """Parses additional exports-file to get all exports aliases and
        meta-query files where they should be saved.
        :return: nothing.
        """
        logging.info('Parse additional export-file %s.' % self.path)

        if not self.check():
            return

        for line in self.get_lines():
            line_fields = line.split()
            if len(line_fields) != 3:
                raise TaskFail('Wrong format in %s: %s.' % (self.path, line))
            (mq_file, export_name, export_alias) = line_fields
            if not self.exports_aliases.get(export_name):
                self.exports_aliases[export_name] = []
            self.exports_aliases[export_name].append({'alias': export_alias,
                                                      'mq_file': mq_file, })
            self.mq_files.add(mq_file)

        logging.info('Export aliases: %s.' % self.exports_aliases)

    def get_aliases(self):
        """Getter for exports aliases.
        :return: dictionary with export aliases.
        """
        return self.exports_aliases

    def get_mq_files(self):
        """Getter for meta-query files.
        :return: set of meta-query files.
        """
        return self.mq_files


class Queries(object):
    """Class to work with query files and queries (exports / procedures)
    themselves.
    """

    def __init__(self, arcadia_dir, meta_subdir, pb_mq_module=None):
        self.meta_dir = os.path.join(arcadia_dir, meta_subdir)
        self.queries_dir = os.path.join(arcadia_dir, cfg.QUERIES_SVN_PATH)
        self.pb_mq_module = pb_mq_module

        self.export_files = set()
        self.procedure_files = set()
        self.export_mq_files = set()
        self.procedure_mq_files = set()

    def load(self):
        """Loads sets of query- and optionally metaquery-files in a proper way
        according to how an object has been created.
        :return: nothing.
        """
        if self.pb_mq_module:
            self.load_mq_lst()
        else:
            self.load_lst()
        self.load_exports_add()

    def load_lst(self):
        """Loads sets of query-files from appropriate lst-files.
        :return: nothing.
        """
        exports_file = self.get_meta_file(cfg.EXPORTS_LST_FILE)
        procedures_file = self.get_meta_file(cfg.PROCEDURES_LST_FILE)

        self.export_files = \
                    LstFile(exports_file, self.queries_dir).get_query_files()
        self.procedure_files = \
                    LstFile(procedures_file, self.queries_dir).get_query_files()

    def load_mq_lst(self):
        """Loads sets of metaquery-files and sets of appropriate query-files
        used by metaquery ones.
        :return: nothing.
        """
        exports_mq_file = self.get_meta_file(cfg.EXPORTS_MQ_LST_FILE)
        procedures_mq_file = self.get_meta_file(cfg.PROCEDURES_MQ_LST_FILE)

        self.export_mq_files = \
                LstFile(exports_mq_file, self.queries_dir).get_query_files()
        self.procedure_mq_files = \
                LstFile(procedures_mq_file, self.queries_dir).get_query_files()

        self.get_prog_from_mq()

    def load_exports_add(self):
        """Loads additional exports file.
        :return: nothing.
        """
        exports_add_file = self.get_meta_file(cfg.EXPORTS_ADD_FILE)

        exports_add = ExportsAddFile(exports_add_file)
        exports_add.parse()
        self.release_add_files = exports_add.get_mq_files()

        self.set_exports(exports_add.get_aliases())

    def get_prog_from_mq(self):
        """Gets sets of query-files (program/condition) from current set of
        metaquery-files.
        :return: nothing.
        """
        logging.info('Get prog-files from mq-files.')
        for mq_file in self.get_all_mq_files():
            mq_dir = os.path.dirname(mq_file)
            mq_file_path = os.path.join(self.queries_dir, mq_file)
            mq_msg = pbmsg.get_mq(self.pb_mq_module, mq_file_path)
            prog_files = metaquery.get_prog_files(mq_msg)
            for export_file in prog_files['export_files']:
                self.export_files.add(os.path.join(mq_dir, export_file))
            for procedure_file in prog_files['procedure_files']:
                self.procedure_files.add(os.path.join(mq_dir, procedure_file))
        logging.info('Export files in mq-files: %s.' % self.export_files)
        logging.info('Procedure files in mq-files: %s.' % self.procedure_files)

    def set_exports(self, exports_aliases):
        """Gets list of export files and makes a list of dictionaries containing
        (name of export, export's aliases, program file, condition file).
        :return: list of dictionaries with information about exports.
        """
        exports_dict = {}

        for export_file in self.get_export_files():
            if export_file.endswith(cfg.QUERY_SFX):
                export_sfx = cfg.QUERY_SFX
            elif export_file.endswith(cfg.CONDITION_SFX):
                export_sfx = cfg.CONDITION_SFX
            else:
                raise TaskFail('Unknown export file: %s.' % export_file)

            export_base_name = os.path.basename(export_file)
            export_name = export_base_name.replace(export_sfx, '', 1)

            if not exports_dict.get(export_name):
                exports_dict[export_name] = {
                    'name': export_name,
                    'aliases': exports_aliases.get(export_name, []),
                }
            exports_dict[export_name][export_sfx] = export_file

        self.exports = exports_dict.values()
        logging.info('Exports: %s.' % self.exports)

    def get_meta_file(self, file_name):
        """Getter for absolute path to meta file @file_name.
        :return: absolute path to @file_name.
        """
        return os.path.join(self.meta_dir, file_name)

    def get_export_files(self):
        """Getter for export files.
        :return: set of export files.
        """
        return self.export_files

    def get_procedure_files(self):
        """Getter for procedure files.
        :return: set of procedure files.
        """
        return self.procedure_files

    def get_all_prog_files(self):
        """Getter for all program files.
        :return: set of all program files.
        """
        return set.union(self.export_files, self.procedure_files)

    def get_all_mq_files(self):
        """Getter for all metaquery files.
        :return: set of all metaquery files.
        """
        return set.union(self.export_mq_files, self.procedure_mq_files)

    def get_exports(self):
        """Getter for exports.
        :return: list of exports.
        """
        return self.exports

    def get_release_add_files(self):
        """Getter for additional release meta-query files.
        :return: set of meta-query files.
        """
        return self.release_add_files

    def install_prog_files(self, dst_root, copy_func):
        """Installs current prog-files into @dst_root using @copy_func.
        :return: nothing.
        """
        logging.info('Install prog-files to %s.' % dst_root)
        for prog_file in self.get_all_prog_files():
            prog_in_file = os.path.join(self.queries_dir, prog_file)
            if not os.path.isfile(prog_in_file):
                raise TaskFail('Input file %s does not exist.' % prog_in_file)
            prog_out_file = os.path.join(dst_root, prog_file)
            make_folder(os.path.dirname(prog_out_file))
            copy_func(prog_in_file, prog_out_file)
        logging.info('All prog-files are installed.')

    def install_mq_files(self, dst_root, triggers_props):
        """Installs current metaquery-files into @dst_root applying properties
        from @triggers_props.
        :return: nothing.
        """
        if not self.pb_mq_module:
            logging.info('Skip installing mq-files (not used).')
            return
        logging.info('Install mq-files to %s.' % dst_root)
        for mq_file in self.get_all_mq_files():
            mq_in_file = os.path.join(self.queries_dir, mq_file)
            if not os.path.isfile(mq_in_file):
                raise TaskFail('Input file %s does not exist.' % mq_in_file)
            mq_msg = pbmsg.get_mq(self.pb_mq_module, mq_in_file)
            mq_msg = metaquery.set_procedure_tag(triggers_props['tag'], mq_msg)
            mq_out_file = os.path.join(dst_root, mq_file)
            make_folder(os.path.dirname(mq_out_file))
            pbmsg.save_common_msg(mq_msg, mq_out_file)
        logging.info('All mq-files are installed.')


class MetaQueryMaker(object):
    """Class for making deployment MetaQuery files.
    """

    def __init__(self, use_pregenerated_mq, deploy_dir):
        self.use_pregenerated_mq = use_pregenerated_mq
        self.deploy_dir = deploy_dir
        self.proc_file = os.path.join(deploy_dir, cfg.MQ_PROC_FILE)
        self.enable_file = os.path.join(deploy_dir, cfg.MQ_ENABLE_FILE)
        self.release_file = os.path.join(deploy_dir, cfg.MQ_RELEASE_FILE)

    def make(self, module_name, triggers_udf_names, queries, triggers_props):
        """Makes metaquery files for query files and queries from @queries
        using tag @triggers_props. Module @module_name is used to work with
        appropriate protobuf-messages.
        :return: nothing.
        """
        module = __import__(module_name, fromlist=['TMetaQuery'])

        if not self.use_pregenerated_mq:
            proc_msg = module.TMetaQuery()

        enable_msg = module.TMetaQuery()
        release_msg = module.TMetaQuery()

        release_add_msgs = {}
        for release_add_file in queries.get_release_add_files():
            release_add_msgs[release_add_file] = module.TMetaQuery()

        triggers_tag = triggers_props['tag']

        for trigger_udf_name in triggers_udf_names:
            if not self.use_pregenerated_mq:
                enable_msg = metaquery.enable_trigger(
                    trigger_udf_name, triggers_tag, enable_msg
                )

            # TODO(rdna@): Uncomment the following code as soon as KiWi is
            #              ready to handle release_trigger-statements. Now it
            #              is not. Ask blinkov@ for details.
            # release_msg = metaquery.release_trigger(trigger_udf_name,
            #        triggers_tag, release_msg)

        for procedure_file in queries.get_procedure_files():
            if not self.use_pregenerated_mq:
                proc_msg = metaquery.add_procedure(procedure_file, triggers_tag, proc_msg)

            enable_msg = metaquery.enable_procedure(procedure_file, triggers_tag, enable_msg)
            release_msg = metaquery.release_procedure(procedure_file, triggers_tag, release_msg)

        if not self.use_pregenerated_mq:
            pbmsg.save_common_msg(proc_msg, self.proc_file)

        pbmsg.save_common_msg(enable_msg, self.enable_file)

        for export in queries.get_exports():
            if not self.use_pregenerated_mq:
                release_msg = metaquery.release_export(export, release_msg)

            release_add_msgs = metaquery.release_add_exports(export, release_add_msgs)

        pbmsg.save_common_msg(release_msg, self.release_file)

        for release_add_file in queries.get_release_add_files():
            release_add_msg = release_add_msgs[release_add_file]
            release_add_file = os.path.join(self.deploy_dir, release_add_file)
            pbmsg.save_common_msg(release_add_msg, release_add_file)
