"""Functions to make meta query messages.

Interface summary:

    import metaquery

    # Add AddUdf-message to mq_msg and fill it with triggers_tag and data from
    # udfs_msg.
    mq_msg = metaquery.add_udf(triggers_tag, udfs_msg, mq_msg)

    # Add information about files in trigger_dir as Files-messages to
    # AddUdf-message in mq_msg for trigger_name.
    mq_msg = metaquery.add_files_to_udf(trigger_name, trigger_dir, mq_msg)
"""

import logging
import os
import re

from sandbox.projects.BuildKiwiTriggers import cfg
from sandbox.projects.BuildKiwiTriggers import util
from sandbox.projects.BuildKiwiTriggers.triggers_data import PathEval
from sandbox.projects.BuildKiwiTriggers.triggers_data import TriggersData


NameType = util.enum(
    TRIGGER_FULL=1,
    PROCEDURE_FULL=2,
    PROCEDURE_BRANCH=3,
    PROCEDURE_RELEASE=4,
)


def tpl2file(tpl_name):
    """Converts metaquery template name @tpl_name to metaquery result file name.
    :return: appropriate metaquery file name.
    """
    # Now we use the same names for templates and generated files. It may be
    # changed in the future.
    return tpl_name


def add_udf(triggers_props, udfs_msg, mq_msg):
    """Adds AddUdf-message to @mq_msg and fills it with @triggers_props and data
    from @udfs_msg.
    :return: @mq_msg after adding AddUdf-message.
    """
    logging.info('Make AddUdf statement for %s UDF.' % udfs_msg.Name)

    versions_msg = udfs_msg.Versions[0]

    add_udf_msg = mq_msg.Statements.add().AddUdf
    add_udf_msg.Name = udfs_msg.Name
    add_udf_msg.Tag = triggers_props['tag']

    init_str = TriggersData.get_init_str(versions_msg.DataPath)
    if init_str:
        add_udf_msg.InitStr = init_str
    elif versions_msg.InitStr:
        add_udf_msg.InitStr = versions_msg.InitStr

    allowed_exec_crash_count = triggers_props.get('AllowedExecCrashCount')
    if allowed_exec_crash_count:
        add_udf_msg.AllowedExecCrashCount = allowed_exec_crash_count

    for params_msg in versions_msg.Params:
        add_udf_msg.Params.add().MergeFrom(params_msg)

    for results_msg in versions_msg.Results:
        add_udf_msg.Results.add().MergeFrom(results_msg)

    is_so_found = False
    for src_files_msg in versions_msg.Files:
        is_so = bool(versions_msg.SoName.endswith(src_files_msg.RelativePath))
        is_locked = bool(src_files_msg.LockInMemory)
        if is_so or is_locked:
            dst_files_msg = add_udf_msg.Files.add()
            dst_files_msg.MergeFrom(src_files_msg)
            if is_so:
                dst_files_msg.IsLibrary = True
                is_so_found = True
            elif is_locked:
                dst_files_msg.RelativePath = PathEval(dst_files_msg.RelativePath).get_path()
    if not is_so_found:
        files_so_msg = add_udf_msg.Files.add()
        files_so_msg.RelativePath = os.path.basename(versions_msg.SoName)
        files_so_msg.IsLibrary = True

    return mq_msg


def get_trigger_udf_names(mq_msg):
    """Gets set of trigger's Udf names from metaquery message.
    :return: set of trigger's Udf names.
    """
    trigger_udf_names = set()
    for stat_msg in mq_msg.Statements:
        if stat_msg.HasField('AddUdf'):
            trigger_udf_names.add(stat_msg.AddUdf.Name)
        elif stat_msg.HasField('DeleteUdf'):
            trigger_udf_names.add(stat_msg.DeleteUdf.Name)
    return trigger_udf_names


def handle_udf_statements(triggers_tag, mq_msg):
    """Handles @mq_msg for trigger @trigger_udf_name: checks for known
    statement-fields and sets valid tag info according to @triggers_tag.
    :return: @mq_msg after handling all the statements in it.
    """
    for stat_msg in mq_msg.Statements:
        if stat_msg.HasField('AddUdf'):
            stat_msg.AddUdf.Tag = triggers_tag
        elif stat_msg.HasField('DeleteUdf'):
            stat_msg.DeleteUdf.Tag = triggers_tag
    return mq_msg


def handle_lock_mask(add_udf_msg):
    """Checks files from @add_udf_msg and finds those which contain lock mask
    in RelativePath. Files-messages with lock mask are removed from
    @add_udf_msg and the mask itself is added to regular explession.
    :return: tuple(add_udf_msg message with lock masks removed,
                   string with lock mask regular explession)
    """
    # FIXME(rdna@): Combining enumerate() and del on repeated messages
    # can lead to deleting wrong message. This should be investigated.
    lock_re = None
    lock_re_set = set()
    for file_i, files_msg in list(enumerate(add_udf_msg.Files)):
        logging.info('Check %s for mask.' % files_msg.RelativePath)
        if files_msg.LockInMemory and '*' in files_msg.RelativePath:
            logging.info('Add %s to lock mask.' % files_msg.RelativePath)
            lock_re_set.add(files_msg.RelativePath)
            del add_udf_msg.Files[file_i - len(lock_re_set) + 1]
    if lock_re_set:
        lock_re = r'(' + r'|'.join(lock_re_set) + r')$'
    logging.info('Lock mask: %s' % lock_re)
    return (add_udf_msg, lock_re)


def add_files_to_udf(trigger_name, trigger_dir, mq_msg):
    """Adds information about files in @trigger_dir as Files-messages to
    corresponding AddUdf-message in @mq_msg for @trigger_name.
    :return: @mq_msg after adding Files-messages.
    """
    for stat_msg in mq_msg.Statements:
        if not stat_msg.HasField('AddUdf'):
            continue

        add_udf_msg = stat_msg.AddUdf
        trigger_datapath = TriggersData.get_datapath(trigger_dir, trigger_name, add_udf_msg.Name)

        (add_udf_msg, lock_re) = handle_lock_mask(add_udf_msg)

        for dirpath, _, filenames in os.walk(trigger_datapath, onerror=True):
            for filename in filenames:
                if filename in cfg.NON_DEPLOYABLE_FILES:
                    continue

                file_path = os.path.join(dirpath, filename)
                file_relpath = file_path.replace(trigger_dir + '/', '', 1)

                is_already_added = False
                for files_msg in add_udf_msg.Files:
                    if file_relpath.endswith(files_msg.RelativePath):
                        files_msg.RelativePath = file_relpath
                        logging.info(
                            'Skip file %s (exists, ends with %s).' % (file_relpath, files_msg.RelativePath)
                        )
                        is_already_added = True
                        break
                if is_already_added:
                    continue

                logging.info('Add file %s for UDF %s.' % (file_relpath, add_udf_msg.Name))
                new_files_msg = add_udf_msg.Files.add()
                new_files_msg.RelativePath = file_relpath

                if lock_re and re.search(lock_re, file_relpath):
                    logging.info('Lock file %s by mask.' % file_relpath)
                    new_files_msg.LockInMemory = True

    return mq_msg


def make_proc_name(name, tag, name_type):
    """Converts name [of trigger or other procedure] @name to procedure name in
    accordance with name type @name_type using tag @tag.
    :return: procedure name.
    """
    # TODO(rdna@): Now KiWi doesn't support ``.'' in procedure names so we
    #              replace it with ``_''.
    #              We should use ``.'' as soon as KiWi supports it.
    return {
        NameType.TRIGGER_FULL: '%s.%s' % (name, tag),
        NameType.PROCEDURE_FULL: '%s_%s' % (name, tag.replace('.', '_')),
        NameType.PROCEDURE_BRANCH: '%s_%s' % (name, tag.rsplit('.', 1)[0]),
        NameType.PROCEDURE_RELEASE: name,
    }[name_type]


def trigger_name2proc(trigger_udf_name, tag, name_type=NameType.TRIGGER_FULL):
    """Converts trigger name @trigger_udf_name to procedure name in accordance
    with name type @name_type using tag @tag.
    :return: procedure name.
    """
    return make_proc_name(trigger_udf_name, tag, name_type)


def procedure_file2name(file_name, tag, name_type=NameType.PROCEDURE_FULL):
    """Converts procedure file name @file_name to procedure name in accordance
    with name type @name_type using tag @tag.
    :return: procedure name.
    """
    base_name = os.path.basename(file_name).replace(cfg.QUERY_SFX, '', 1)
    return make_proc_name(base_name, tag, name_type)


def add_procedure(procedure_file, triggers_tag, mq_msg):
    """Adds information about procedure from @procedure_file to corresponding
    AddProcedure-message in @mq_msg using tag @triggers_tag.
    :return: @mq_msg after adding AddProcedure-messages.
    """
    logging.info('Make AddProcedure statement for file %s.' % procedure_file)
    add_proc_msg = mq_msg.Statements.add().AddProcedure

    add_proc_msg.Name = procedure_file2name(procedure_file, triggers_tag)
    add_proc_msg.ProgramFile = procedure_file

    return mq_msg


def change_procedure(common_name, specific_name, mq_msg):
    """Adds statements to TMetaQuery-message @mq_msg to switch procedure with
    name @common_name to a new specific name @specific_name.
    :return: @mq_msg after adding DeleteProcedure- / AddProcedure- messages.
    """
    logging.info('Change procedure: %s -> %s.' % (common_name, specific_name))
    del_proc_msg = mq_msg.Statements.add().DeleteProcedure
    del_proc_msg.Name = common_name

    add_proc_msg = mq_msg.Statements.add().AddProcedure
    add_proc_msg.Name = common_name
    add_proc_msg.Alias = specific_name

    return mq_msg


def get_prog_files(mq_msg):
    """Gets program- and conditional- file names from AddProcedure- / AddExport-
    statements of TMetaQuery-message @mq_msg.
    :return: dictionary with ``export_files'' and ``procedure_files'' keys.
    Values are lists of appropriate files.
    """
    prog_files = {
        'export_files': [],
        'procedure_files': [],
    }

    for stat_msg in mq_msg.Statements:
        if stat_msg.HasField('AddProcedure'):
            add_procedure_msg = stat_msg.AddProcedure
            if add_procedure_msg.HasField('ProgramFile'):
                prog_files['procedure_files'].append(add_procedure_msg.ProgramFile)
        elif stat_msg.HasField('AddExport'):
            add_export_msg = stat_msg.AddExport
            if add_export_msg.HasField('ConditionFile'):
                prog_files['export_files'].append(add_export_msg.ConditionFile)
            if add_export_msg.HasField('ProgramFile'):
                prog_files['export_files'].append(add_export_msg.ProgramFile)

    return prog_files


def set_procedure_tag(triggers_tag, mq_msg):
    """Changes AddProcedure- / DeleteProcedure- statements in TMetaQuery-message
    @mq_msg so that Name in every one contains triggers' tag @triggers_tag.
    :return: @mq_msg after changing AddProcedure- / DeleteProcedure- messages.
    """
    prog_files = get_prog_files(mq_msg)
    if not prog_files['procedure_files']:
        logging.info('Skip message (procedure is not found: %s).' % prog_files)
        return mq_msg
    procedure_file = prog_files['procedure_files'][0]
    procedure_name = procedure_file2name(procedure_file, triggers_tag)
    logging.info('Change procedure name to %s.' % procedure_name)
    for stat_msg in mq_msg.Statements:
        if stat_msg.HasField('DeleteProcedure'):
            stat_msg.DeleteProcedure.Name = procedure_name
        elif stat_msg.HasField('AddProcedure'):
            stat_msg.AddProcedure.Name = procedure_name
    return mq_msg


def enable_trigger(trigger_udf_name, triggers_tag, mq_msg):
    """Adds statements to TMetaQuery-message @mq_msg to enable new trigger
    @trigger_udf_name with tag @triggers_tag.
    :return: @mq_msg after adding DeleteProcedure- / AddProcedure- messages.
    """
    branch_name = trigger_name2proc(
        trigger_udf_name, triggers_tag,
        name_type=NameType.PROCEDURE_BRANCH
    )
    full_name = trigger_name2proc(trigger_udf_name, triggers_tag)
    return change_procedure(branch_name, full_name, mq_msg)


def release_trigger(trigger_udf_name, triggers_tag, mq_msg):
    """Adds statements to TMetaQuery-message @mq_msg to release new trigger
    @trigger_udf_name with tag @triggers_tag.
    :return: @mq_msg after adding DeleteProcedure- / AddProcedure- messages.
    """
    branch_name = trigger_name2proc(
        trigger_udf_name, triggers_tag,
        name_type=NameType.PROCEDURE_BRANCH
    )
    release_name = trigger_name2proc(
        trigger_udf_name, triggers_tag,
        name_type=NameType.PROCEDURE_RELEASE
    )
    return change_procedure(release_name, branch_name, mq_msg)


def enable_procedure(procedure_file, triggers_tag, mq_msg):
    """Adds statements to TMetaQuery-message @mq_msg to enable new procedure
    with tag @triggers_tag from procedure file @procedure_file.
    :return: @mq_msg after adding DeleteProcedure- / AddProcedure- messages.
    """
    branch_name = procedure_file2name(
        procedure_file, triggers_tag,
        name_type=NameType.PROCEDURE_BRANCH
    )
    full_name = procedure_file2name(procedure_file, triggers_tag)
    return change_procedure(branch_name, full_name, mq_msg)


def release_procedure(procedure_file, triggers_tag, mq_msg):
    """Adds statements to TMetaQuery-message @mq_msg to release new procedure
    with tag @triggers_tag from procedure file @procedure_file.
    :return: @mq_msg after adding DeleteProcedure- / AddProcedure- messages.
    """
    branch_name = procedure_file2name(
        procedure_file, triggers_tag,
        name_type=NameType.PROCEDURE_BRANCH
    )
    release_name = procedure_file2name(
        procedure_file, triggers_tag,
        name_type=NameType.PROCEDURE_RELEASE
    )
    return change_procedure(release_name, branch_name, mq_msg)


def release_export(export, mq_msg):
    """Adds ChangeExport-message to @mq_msg and fills it with data from
    @export.
    :return: @mq_msg after adding ChangeExport-message.
    """
    logging.info('Make ChangeExport statement for export %s.' % export['name'])
    ch_exp_msg = mq_msg.Statements.add().ChangeExport

    ch_exp_msg.Name = export['name']
    ch_exp_msg.ProgramFile = export[cfg.QUERY_SFX]
    ch_exp_msg.ConditionFile = export[cfg.CONDITION_SFX]

    return mq_msg


def release_add_exports(export, mq_msgs):
    """Adds ChangeExport-messages for @export's aliases to appropriate members
    of @mq_msgs. ``Export's alias'' means usual export but with different name
    and saved in different mq-file to release export on additional clusters
    (e.g. X-cluster).
    :return: @mq_msgs after adding ChangeExport-messages.
    """
    for export_alias in export['aliases']:
        logging.info('Make ChangeExport statements for export alias %s.' % export_alias)
        st_msg = mq_msgs[export_alias['mq_file']].Statements.add()
        ch_exp_msg = st_msg.ChangeExport
        ch_exp_msg.Name = export_alias['alias']
        ch_exp_msg.ProgramFile = export[cfg.QUERY_SFX]
        ch_exp_msg.ConditionFile = export[cfg.CONDITION_SFX]
    return mq_msgs
