"""Interface to work with prototext files.
"""

import logging
import os

from google.protobuf.text_format import Merge as pb_txt_merge
from google.protobuf.text_format import MessageToString as pb_txt_msg2str

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects.BuildKiwiTriggers import cfg


def get_common_msg(module_name, msg_name, file_name):
    """If @file_name is set, gets all the text from it and fills protobuf
    message @msg_name with it, otherwise just makes empty message. Module
    @module_name is used to work with the message.
    :return: protobuf message of type @msg_name.
    """
    module = __import__(module_name, fromlist=[msg_name])
    if msg_name == 'TMeta':
        msg = module.TMeta()
    elif msg_name == 'TMetaQuery':
        msg = module.TMetaQuery()
    else:
        logging.info('Unknown message name: %s.' % msg_name)
        return None

    if not file_name:
        logging.info('Make empty %s message.' % msg_name)
        return msg

    logging.info('Parse proto-text file %s.' % file_name)
    with open(file_name) as file_fd:
        file_str = file_fd.read()
    pb_txt_merge(file_str, msg)
    logging.info('PB message from %s:\n%s\n' % (file_name, pb_txt_msg2str(msg)))
    return msg


def get_meta(module_name, file_name=None):
    """Wrapper for get_common_msg() function to work with TMeta messages.
    :return: TMeta message.
    """
    return get_common_msg(module_name, 'TMeta', file_name)


def get_mq(module_name, file_name=None):
    """Wrapper for get_common_msg() function to work with TMetaQuery messages.
    :return: TMetaQuery message.
    """
    return get_common_msg(module_name, 'TMetaQuery', file_name)


def save_common_msg(msg, file_name):
    """Saves protobuf message @msg to file @file_name.
    :return: nothing.
    """
    logging.info('Save proto-text to file %s.' % file_name)
    with open(file_name, 'w') as file_fd:
        file_fd.write(pb_txt_msg2str(msg))


def save_all(resource_path, trigger_udf, meta_msg, mq_msg):
    """Saves protobuf messages @meta_msg and @mq_msg to appropriate files for
    trigger @trigger_udf in resource directory @resource_path.
    :return: nothing.
    """
    trigger_dir = os.path.join(resource_path, trigger_udf)
    make_folder(trigger_dir)

    trigger_meta_file = os.path.join(trigger_dir, cfg.PB_META_FILE)
    trigger_mq_file = os.path.join(trigger_dir, cfg.MQ_META_FILE)

    save_common_msg(meta_msg, trigger_meta_file)
    save_common_msg(mq_msg, trigger_mq_file)


def mk_meta_file(module_name, in_dir, out_file, udf_id_min=(1 << 15)):
    """Find meta data files in @in_dir and aggregate them into @out_file
    changing UdfId to avoid intersection. UdfId-s are generated sequentially
    starting from @udf_id_min.
    :return: resulting @out_file.
    """
    logging.info('Prepare metadata file.')

    triggers_meta_str = ''
    for dirpath, _, filenames in os.walk(in_dir, followlinks=True, onerror=True):
        if cfg.PB_META_FILE not in filenames:
            continue
        trigger_meta_file = os.path.join(dirpath, cfg.PB_META_FILE)
        logging.info('Get meta info from %s.' % trigger_meta_file)
        with open(trigger_meta_file) as trigger_meta_fd:
            trigger_meta_str = trigger_meta_fd.read()
        triggers_meta_str += trigger_meta_str

    meta_module = __import__(module_name, fromlist=['TMeta'])
    meta_msg = meta_module.TMeta()
    pb_txt_merge(triggers_meta_str, meta_msg)

    meta_msg_Udfs_count = len(meta_msg.Udfs)
    if meta_msg_Udfs_count < 1:
        raise TaskFail(
            'There have to be at least one Udfs section (%d given).' % meta_msg_Udfs_count
        )

    for udf_id_shift, udfs_msg in enumerate(meta_msg.Udfs):
        udfs_msg.UdfId = udf_id_min + udf_id_shift

    with open(out_file, 'w') as out_file_fd:
        out_file_fd.write(pb_txt_msg2str(meta_msg))

    return out_file
