"""Post copying hooks.

Interface summary:

    from post_cp_hooks import PostCpHook

    h = PostCpHook(hook_name, path)     # create post hook object
    h.run()                             # run hook
    h.is_cachable()                     # check whether path is cachable or not
                                        # after h has finished
"""

import gzip
import logging
import os

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail

GZ_SUFFIX = '.gz'
GZ_READ_BLOCK = 1 << 12     # bytes


def gunzip_img_models(models_dir):
    """ Find gzip files in @models_dir and gunzip them.
    """
    for dirpath, _, filenames in os.walk(models_dir, onerror=True):
        for filename in filenames:
            if not filename.endswith(GZ_SUFFIX):
                continue
            gz_path = os.path.join(dirpath, filename)
            logging.info('Gunzip %s.' % gz_path)
            gz_fd = gzip.open(gz_path)
            out_path = gz_path[:-len(GZ_SUFFIX)]
            with open(out_path, 'wb') as out_fd:
                while True:
                    chunk = gz_fd.read(GZ_READ_BLOCK)
                    if not chunk:
                        break
                    out_fd.write(chunk)
            gz_fd.close()
            os.unlink(gz_path)


HOOK_MAP = {
    'gunzip_img_models': {
        'func': gunzip_img_models,
        'cachable': False,
    },
}


class PostCpHook(object):
    """Class to work with post copying hooks.
    """

    def __init__(self, name, path):
        self.name = name
        self.path = path
        if not HOOK_MAP.get(self.name):
            raise TaskFail('Unknown hook %s.' % self.name)
        self.func = HOOK_MAP[self.name]['func']
        self.cachable = HOOK_MAP[self.name].get('cachable', False)

    def run(self):
        """Run appropriate hook.
        """
        logging.info('Run %s hook for %s.' % (self.name, self.path))
        self.func(self.path)

    def is_cachable(self):
        """Show whether appropriate path is cachable or not after hook has
        finished. ``Cachable'' means that path and appropriate data have not
        been changed and can be used for other triggers.
        @return: True or False
        """
        return self.cachable
