#!/skynet/python/bin/python

import argparse
import atexit
import copy
import json
import logging
import os
import pprint
import re
import shutil
import subprocess
import sys
import tempfile

from google.protobuf.text_format import Merge as pb_txt_merge
from google.protobuf.text_format import MessageToString as pb_txt_msg2str

from data_cfg import triggers_data

YA_SUBPATH = 'ya'

PROTOC_SUBPATH = 'contrib/tools/protoc'
PROTOC_PROGRAM = 'protoc'

PROTO_SRC_SFX = '.proto'
PROTO_DST_SFX = '_pb2'

MQ_PROTO_FILE = 'yweb/robot/kiwi/protos/metaquery.proto'
MQ_MESSAGE = 'TMetaQuery'

CMAKE_LISTS_FILE = 'CMakeLists.txt'

DEPLOY_MQ_FILE = 'deploy.mq'
DEPLOY_TAG = "trunk.HEAD"

DATA_CFG_FILE = 'data.json'

TRIGGERS_SUBDIRS = ['yweb/robot/kiwi_queries']


def err_handler(err):
    raise Exception(err)


def get_args():
    parser = argparse.ArgumentParser(description='Splits data_cfg.py.')
    parser.add_argument('--arcadia-root', metavar='path', action='store',
                        type=str, dest='arcadia_root',
                        help='path to arcadia root directory')
    parser.add_argument('--make-json', action='store_true', dest='make_json',
                        default=False,
                        help='save result into per-trigger data.json')
    return parser.parse_args()


def find_arcadia_root():
    cur_path = os.path.realpath(__file__)
    while cur_path != '/':
        if os.path.basename(cur_path) == 'arcadia':
            return cur_path
        cur_path = os.path.dirname(cur_path)
    raise Exception('Cannot find arcadia root. Use --arcadia-root option.')


def conver_scheme(data_str):
    data_str = data_str.replace('arc_root://', 'svn_branch_root://')
    data_str = data_str.replace('wc_file://', 'svn_branch_root://')
    return data_str


def mk_json(trigger_dir, trigger_udf, trigger_data):
    trigger_data['WARNING!!!'] = [
        'This file is generated automatically for ``%s\'\' from' % trigger_udf,
        'trunk/arcadia/sandbox/projects/BuildKiwiTriggers/data_cfg.py',
        'and intended for experimental using.',
        'You should use it nowhere.', ]
    json_cfg_str = json.dumps(trigger_data, indent=4, sort_keys=True)
    json_cfg_str = conver_scheme(json_cfg_str)
    json_cfg_path = os.path.join(trigger_dir, DATA_CFG_FILE)
    with open(json_cfg_path, 'w') as json_cfg_fd:
        json_cfg_fd.write(json_cfg_str)


def get_protoc(arcadia_root):
    ya_path = os.path.join(arcadia_root, YA_SUBPATH)
    protoc_src_path = os.path.join(arcadia_root, PROTOC_SUBPATH)
    ya_cmd = '%s build -j20 -C %s' % (ya_path, protoc_src_path)
    subprocess.check_output(ya_cmd.split())
    protoc_path = os.path.join(protoc_src_path, PROTOC_PROGRAM)
    return protoc_path


def pb_path_to_module(pb_path):
    return pb_path.replace(PROTO_SRC_SFX, PROTO_DST_SFX).replace(r'/', '.')


def get_proto_set(arcadia_root, proto_file, result=set()):
    result.add(proto_file)
    proto_path = os.path.join(arcadia_root, proto_file)
    if not os.path.isfile(proto_path):
        raise Exception('Proto file %s does not exist.' % proto_path)
    with open(proto_path) as proto_fd:
        proto_data = proto_fd.read()
    import_re = re.compile(r'^\s*import\s*"(?P<file>\S+)"\s*;\s*$', re.M)
    for search_res in import_re.finditer(proto_data):
        import_file = search_res.group('file')
        get_proto_set(arcadia_root, import_file, result)
    return result


def get_mq_module(arcadia_root, proto_main_file):
    protoc_path = get_protoc(arcadia_root)

    proto_files = get_proto_set(arcadia_root, proto_main_file)
    proto_paths = [os.path.join(arcadia_root, pf) for pf in proto_files]

    out_dir = tempfile.mkdtemp(prefix='protos.')
    atexit.register(shutil.rmtree, out_dir)
    sys.path.append(out_dir)

    protoc_cmd = '%(cmd)s -I%(inc_root)s --python_out=%(out_dir)s %(files)s' % {
        'cmd': protoc_path,
        'inc_root': arcadia_root,
        'out_dir': out_dir,
        'files': ' '.join(proto_paths),
    }
    subprocess.check_output(protoc_cmd.split())

    for dirpath, _, _ in os.walk(out_dir, onerror=err_handler):
        with open(os.path.join(dirpath, '__init__.py'), 'w'):
            pass

    mq_proto_module = pb_path_to_module(MQ_PROTO_FILE)
    return __import__(mq_proto_module, fromlist=[MQ_MESSAGE])


def is_lib(file_name):
    if file_name.startswith('lib') and file_name.endswith('.so'):
        return True
    return False


def get_files(trigger_data):
    files = {}
    for data_item in trigger_data['files']:
        src = conver_scheme(data_item['src'])
        dst = data_item.get('dst', os.path.basename(src))
        local_path = None
        if src.endswith('/'):
            uri = src
            local_path = 'data'
            relative_path = '.'
        else:
            uri = src
            relative_path = dst
        if not files.get(relative_path):
            files[relative_path] = {'URI': uri}
            if local_path:
                files[relative_path]['LocalPath'] = local_path
        else:
            raise Exception('RelativePath duplicates in %s.' % trigger_data)
    return files


def merge_files_to_add_udf(files, add_udf_msg, deploy_mq):
    for file_msg in add_udf_msg.Files:
        if file_msg.HasField('RelativePath'):
            relative_path = file_msg.RelativePath
        elif file_msg.HasField('URI'):
            relative_path = os.path.basename(file_msg.URI)
        else:
            raise Exception('Files does not have neither RelativePath nor ' +
                            'URI: %s.' % deploy_mq)

        if is_lib(relative_path) and not file_msg.HasField('IsLibrary'):
            logging.warn('Fix IsLibrary in %s.' % deploy_mq)
            file_msg.IsLibrary = True

        if files.get(relative_path):
            file_item = files.pop(relative_path)
            file_msg.URI = file_item['URI']
            if file_item.get('LocalPath'):
                file_msg.LocalPath = file_item['LocalPath']

        if file_msg.HasField('RelativePath') and file_msg.HasField('URI') and \
                file_msg.RelativePath == os.path.basename(file_msg.URI):
            file_msg.ClearField('RelativePath')

    return copy.deepcopy(files)


def add_files_to_add_udf(files, add_udf_msg):
    for relative_path, file_item in files.iteritems():
        file_msg = add_udf_msg.Files.add()
        file_msg.URI = file_item['URI']
        file_msg.RelativePath = relative_path
        if file_item.get('LocalPath'):
            file_msg.LocalPath = file_item['LocalPath']


def ignore_space_changes(file_name):
    def get_out(cmd):
        return subprocess.check_output(cmd.split())

    if not get_out('svn di %s' % file_name):
        return
    if not get_out('svn di -x -w %s' % file_name):
        # There are only space changes, ignore it.
        subprocess.check_call(['svn', 'revert', file_name])


def add_data_to_deploy(mq_module, deploy_mq, trigger_data):
    mq_msg = mq_module.TMetaQuery()
    with open(deploy_mq) as deploy_fd:
        pb_txt_merge(deploy_fd.read(), mq_msg)

    for stat_msg in mq_msg.Statements:
        if stat_msg.HasField('AddUdf'):
            files = get_files(trigger_data)
            add_udf_msg = stat_msg.AddUdf
            add_udf_msg.Tag = DEPLOY_TAG
            rest_files = merge_files_to_add_udf(files, add_udf_msg, deploy_mq)
            add_files_to_add_udf(rest_files, add_udf_msg)
        elif stat_msg.HasField('DeleteUdf'):
            delete_udf_msg = stat_msg.DeleteUdf
            delete_udf_msg.Tag = DEPLOY_TAG

    with open(deploy_mq, 'w') as deploy_fd:
        deploy_fd.write(pb_txt_msg2str(mq_msg))

    ignore_space_changes(deploy_mq)


def main():
    args = get_args()

    arcadia_root = args.arcadia_root
    if not arcadia_root:
        # Try to find it by ourselves.
        arcadia_root = find_arcadia_root()

    mq_module = get_mq_module(arcadia_root, MQ_PROTO_FILE)

    udf_re = re.compile(r'^UDF\((?P<udf_name>[^\)]+)\)', flags=re.M)

    for triggers_subdir in TRIGGERS_SUBDIRS:
        triggers_dir = os.path.join(arcadia_root, triggers_subdir)
        for dir_root, _, file_names in os.walk(triggers_dir,
                                               onerror=err_handler):
            for file_name in file_names:
                if file_name != CMAKE_LISTS_FILE:
                    continue
                cmake_path = os.path.join(dir_root, file_name)
                with open(cmake_path) as cmake_fd:
                    cmake_str = cmake_fd.read()
                search_res = udf_re.search(cmake_str)
                if not search_res:
                    continue
                trigger_udf = os.path.basename(dir_root)
                trigger_data = triggers_data.pop(trigger_udf, None)
                if not trigger_data:
                    continue
                if args.make_json:
                    mk_json(dir_root, trigger_udf, trigger_data)
                    continue
                deploy_mq = os.path.join(dir_root, DEPLOY_MQ_FILE)
                if not os.path.exists(deploy_mq):
                    logging.warn('%s is not found.' % deploy_mq)
                    continue
                add_data_to_deploy(mq_module, deploy_mq, trigger_data)

    logging.warn('===> Unknown triggers:\n%s' % pprint.pformat(triggers_data))


if __name__ == '__main__':
    main()
