"""Classes to encapsulate triggers data configuration.

Interface summary:

    from triggers_data import TriggersData as td

    d = td.get_datapath(d, n)               # get data path for trigger n with
                                            # library directory d

    d = td.get_datapath_str(d, n, q, s)     # get DataPath string for trigger n
                                            # with Udfs.Name q and library
                                            # directory d using "raw" DataPath
                                            # string s

    b = td.get_basedir(d, n, q)             # get BaseDir string for trigger n
                                            # with Udfs.Name q and library
                                            # directory d

    f = td.get_files(n)                     # get files dict for trigger n

    td.is_shared(n)                         # whether trigger n has shared data
                                            # or not

    n = td.names()                          # get triggers names which have data
"""

import os
import re
import logging

from sandbox.projects import resource_types as rt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail

from sandbox.projects.BuildKiwiTriggers.data_cfg import SHARED_DATA_DIR
from sandbox.projects.BuildKiwiTriggers.data_cfg import triggers_data
from sandbox.projects.BuildKiwiTriggers.data_cfg import triggers_datapaths

from sandbox.projects.common import apihelpers


Revision_re = re.compile(r'^r\d+$')
Variable_re = re.compile(r'<(?P<variable>[^>]+)>')


class PathEval(object):
    """Class for evaluating strings (mostly paths) containing specific
    variables. It also has caches to keep results of heavy operations.
    """

    paths_cache = {}
    values_cache = {}

    def __init__(self, path):
        self.path = path
        self.variable = None

    def get_path(self):
        """Get evaluated path for current instance. If path is found in cache,
        appropriate entry will be returned.
        @return: evaluated path
        """
        logging.info('Get path %s ...' % self.path)
        if PathEval.paths_cache.get(self.path):
            logging.info('Cache hit (path): %s.' % self.path)
            return PathEval.paths_cache[self.path]
        search_res = Variable_re.search(self.path)
        if not search_res:
            PathEval.paths_cache[self.path] = self.path
            return PathEval.paths_cache[self.path]
        self.variable = search_res.group('variable')
        value = self.get_value(self.variable)
        eval_path = Variable_re.sub(value, self.path)
        PathEval.paths_cache[self.path] = eval_path
        return PathEval.paths_cache[self.path]

    def get_value(self, variable=None):
        """Evaluate variable @variable or class variable @variable for current
        instance. If variable is found in cache, appropriate entry will be
        returned.
        @return: value that correspond @variable.
        """
        if not self.variable:
            self.variable = variable
        logging.info('Get variable %s ...' % self.variable)
        if PathEval.values_cache.get(self.variable):
            logging.info('Cache hit (variable): %s.' % self.variable)
            return PathEval.values_cache[self.variable]
        if self.variable == 'revision':
            value = self.get_antispam_rules_rev()
        else:
            raise TaskFail('Unknown variable %s.' % self.variable)
        PathEval.values_cache[self.variable] = value
        return PathEval.values_cache[self.variable]

    def get_antispam_rules_rev(self):
        """Get current revision for antispam rules.
        @return: revision in format "rX" where "X" is number.
        """
        logging.info('Get %s for antispam rules.' % self.variable)
        resource = apihelpers.get_last_resource(rt.ANTISPAM_KIWI_RULES)
        if not resource:
            raise TaskFail('Cannot find resource %s.' % rt.ANTISPAM_KIWI_RULES)
        revision = resource.attributes.get(self.variable)
        if not revision:
            raise TaskFail('Cannot get %s for antispam rules.' % self.variable)
        logging.info('Antispam rules %s: %s.' % (self.variable, revision))
        return revision


class TriggersData(object):
    """Class to encapsulate triggers data configuration.
    """

    @staticmethod
    def get_datapath(trigger_dir, trigger_name, trigger_qsname=None, relative=False):
        """Get data path for trigger @trigger_name with library directory
        @trigger_dir. If trigger has different data paths for different Udfs,
        then @trigger_qsname should be passed.
        @return: relative or absolute data path according to @relative.
        """
        trigger_datapath = trigger_dir
        if (
            trigger_name in triggers_data and
            'shared_data' in triggers_data[trigger_name]
        ):
            trigger_datapath = os.path.join(
                trigger_datapath, SHARED_DATA_DIR,
                triggers_data[trigger_name]['shared_data']
            )

        if (
            trigger_name in triggers_data and
            'data_subpath' in triggers_data[trigger_name]
        ):
            trigger_datapath = os.path.join(
                trigger_datapath,
                triggers_data[trigger_name]['data_subpath']
            )

        if trigger_qsname in triggers_datapaths:
            trigger_datapath = os.path.join(
                trigger_datapath,
                triggers_datapaths[trigger_qsname]
            )

        trigger_datapath = PathEval(trigger_datapath).get_path()
        if relative:
            trigger_datapath = os.path.relpath(trigger_datapath)
        return trigger_datapath

    @staticmethod
    def get_datapath_str(trigger_dir, trigger_name, trigger_qsname, raw_str):
        """Get DataPath string for trigger @trigger_name with library directory
        @trigger_dir and Udfs.Name @trigger_qsname. The string is made from
        initial version @raw_str.
        @return: DataPath string that can be used in meta data files.
        """
        # FIXME DataPath is actually used as a string that can contain any
        #       additional trigger parameters delimited by some symbol. So we
        #       have to keep all these possible parameters.
        #       This is hack and should be fixed in the future somehow.
        #       X-Jira: KIWI-268
        trigger_datapath = TriggersData.get_datapath(trigger_dir, trigger_name, trigger_qsname, relative=True)
        str_parts = raw_str.split(';')
        str_parts[0] = trigger_datapath
        for i, str_part in enumerate(str_parts):
            if str_part.startswith('revision='):
                str_parts[i] = 'revision=' + PathEval(trigger_datapath).get_value('revision')
                break
        datapath_str = ';'.join(str_parts)
        return datapath_str

    @staticmethod
    def get_init_str(datapath_str):
        """Get trigger's initialization parameters from @datapath_str.
        @return: string which can be used as InitStr for a trigger.
        """
        return ';'.join(datapath_str.split(';')[1:])

    @staticmethod
    def get_basedir(trigger_dir, trigger_name, trigger_qsname):
        """Get BaseDir string for trigger @trigger_name with library directory
        @trigger_dir and Udfs.Name @trigger_qsname.
        @return: BaseDir string that can be used in meta data files.
        """
        return TriggersData.get_datapath(trigger_dir, trigger_name, trigger_qsname, relative=True)

    @staticmethod
    def get_files(trigger_name):
        """Get files dictionary for specified @trigger_name.
        @return: the dictionary.
        """
        if trigger_name not in triggers_data:
            return None
        files = []
        for file_item in triggers_data[trigger_name].get('files', []):
            file_item['src'] = PathEval(file_item['src']).get_path()
            if file_item.get('dst'):
                file_item['dst'] = PathEval(file_item['dst']).get_path()
            files.append(file_item)
        return files

    @staticmethod
    def is_shared(trigger_name):
        """Determine whether trigger @trigger_name has shared data or not.
        @return: True or False
        """
        if trigger_name not in triggers_data:
            return None
        return 'shared_data' in triggers_data[trigger_name]

    @staticmethod
    def get_shared_data_dir():
        """Get base name of shared data directory.
        @return: directory name
        """
        return os.path.basename(SHARED_DATA_DIR)

    @staticmethod
    def names():
        """Get names of all the triggers which have data files.
        @return: list of trigger names
        """
        # TODO: Sort ``triggers_data'' so that getting files from working copy
        #       happens in the last turn not to block data-thread.
        return triggers_data.keys()

# vim:set ts=4 sw=4 et:
