"""Utility functions which do not operate with triggers directly."""

import logging
import os
import random
import re
import subprocess
import time
import traceback

from sandbox.projects.common.utils import get_short_branch_name, get_svn_info, svn_revision

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.svn import Arcadia


def enum(**enums):
    """
        Simulate enum from C. Create a new class where keys from @enums
        dictionary became class variables with values equal values from @enums.
        @return: created class
    """
    return type('Enum', (object, ), enums)


def check_trigger_file(trigger_name, file_name):
    """Checks that trigger @trigger_name has file @file_name and fails the task
    if file doesn't exist.
    :return: nothing.
    """
    file_basename = os.path.basename(file_name)
    if not os.path.isfile(file_name):
        raise TaskFail('Trigger %s does not have %s.' % (trigger_name, file_basename))


def symlink_rel(src, dst):
    """
        Creates relative symlink @dst -> @src.
        :return: nothing.
    """
    src_rel = os.path.relpath(src, start=os.path.dirname(dst))
    os.symlink(src_rel, dst)


def fast_copy_entry(src, dst, copy_function):
    """
        Copies @src to @dst using @copy_function for every file.
        :return: nothing.
    """
    logging.info('Fast copy %s to %s.' % (src, dst))
    if os.path.isfile(src):
        copy_function(src, dst)
    else:
        copy_path(src, dst, copy_function=copy_function)


def fast_copy(src, dst, copy_function=os.link):
    """
        Copies @src to @dst taking into account ``/'' at the end. For real
        copying fast_copy_entry() is used.
        :return: nothing.
    """
    if src.endswith('/'):
        for entry_name in os.listdir(src):
            entry_src = os.path.join(src, entry_name)
            entry_dst = os.path.join(dst, entry_name)
            fast_copy_entry(entry_src, entry_dst, copy_function)
    else:
        fast_copy_entry(src, dst, copy_function)


def svn_find_uniq_dirs(svn_url, depth='immediates'):
    # TODO: Document the function
    ls_str = Arcadia.list(svn_url, depth=depth)
    dirs_set = set()
    dups_set = set()
    for dirent in ls_str.split('\n'):
        if dirent.endswith('/'):
            dir_leave = dirent.split('/')[-2]
            if dir_leave in dirs_set:
                dups_set.add(dir_leave)
            else:
                dirs_set.add(dir_leave)
    dirs_list = list(dirs_set - dups_set)
    dirs_list.sort()
    return dirs_list


def normalize_svn_url(svn_url):
    # TODO: Document the function
    logging.info('Normalize SVN URL %s ...' % svn_url)
    parsed_url = Arcadia.parse_url(svn_url)
    revision = parsed_url.revision
    if revision:
        revision = revision.replace('r', '')
        if revision == 'HEAD':
            # We should remember certain revision for whole the task
            revision = get_svn_info(svn_url)['entry_revision']
    svn_url = Arcadia.replace(svn_url, revision=revision)
    logging.info('Normalized SVN URL: %s.' % svn_url)
    return svn_url


def get_arc_path(
    arc_root, arc_subpath, local_path=None, use_branch_root=False,
    isfile=False
):
    # TODO: Document the function
    parsed_url = Arcadia.parse_url(arc_root)
    path = parsed_url.path
    if use_branch_root:
        path = path.replace('/arcadia', '/', 1)
    svn_url = Arcadia.replace(arc_root, path=os.path.join(path, arc_subpath))
    if not local_path:
        local_path = arc_subpath
    if os.path.exists(local_path):
        logging.info('The path %s has already been downloaded.' % local_path)
        return os.path.abspath(local_path)
    if isfile:
        make_folder(os.path.dirname(local_path))
        Arcadia.export(svn_url, local_path)
        return os.path.abspath(local_path)
    else:
        return Arcadia.checkout(svn_url, local_path)


def get_branch_name(svn_url):
    """
        Get full branch name from @svn_url or raise exception.
        @returns: full branch name
    """
    logging.info('Get the branch name from svn url %s' % svn_url)
    search_result = re.search(r'arc/(.*)/arcadia', svn_url)
    if search_result:
        branch_name = search_result.group(1)
    else:
        raise TaskFail('Cannot get branch name from %s' % svn_url)
    return branch_name


def get_normalized_branch_name(svn_url):
    """
        Get normalized short branch name from @svn_url.
        @returns: normalized short branch name
    """
    tmp_branch_name = get_short_branch_name(svn_url).replace('-', '_')
    if not tmp_branch_name:
        raise TaskFail('Cannot get short branch name from %s.' % svn_url)
    norm_branch_name = re.sub(r'\W+', '', tmp_branch_name)
    logging.info('Normalized branch name: %s' % norm_branch_name)
    return norm_branch_name


def get_branch_group(svn_url):
    """Get branch group name from @svn_url. Group means the next path component
    after the ``/branches/'' one.
    :return: string with branch group if it is found, None otherwise.
    """
    logging.info('Get branch group name from svn url %s' % svn_url)
    search_res = re.search(r'/+branches/+(?P<branch_group>[^/]+)/+.*/+arcadia',
                              svn_url)
    if not search_res:
        return None
    branch_group = search_res.group('branch_group')
    logging.info('Branch group name: %s' % branch_group)
    return branch_group


def get_revision(svn_url):
    """
        Get revision part from @svn_url.
        @returns: revision
    """
    # We don't use svn_revision() here to avoid request to svn server.
    return Arcadia.parse_url(svn_url).revision


def rsync_ls(rsync_url, con_timeout=30, io_timeout=30, tries=3, try_time=15):
    """
        Get listing of rsync URL @rsync_url. Try to get listing @tries times
        witn @try_time interval between tries.
        @return: string with listing.
    """
    rsync_ls_cmd = 'rsync --contimeout=%d --timeout=%d --list-only %s' % \
            (con_timeout, io_timeout, rsync_url)
    rsync_ls_args = rsync_ls_cmd.split()
    while (tries > 0):
        try:
            out_str = subprocess.check_output(rsync_ls_args)
            return out_str
        except subprocess.CalledProcessError as e:
            logging.exception(e)
            tries -= 1
            if tries > 0:
                time.sleep(try_time)
    raise TaskFail('Cannot get listing of %s.' % rsync_url)


def create_robot_tag(svn_url, rev):
    # TODO: Document the function
    logging.info('Create robot svn tag from %s.' % svn_url)
    short_branch_name = get_short_branch_name(svn_url)
    if not short_branch_name:
        raise TaskFail('Cannot get short branch name from %s.' % svn_url)
    robot_tag_path = 'arc/tags/robot/%s_rev%s' % (short_branch_name, rev)
    tag_svn_url = Arcadia.replace(svn_url, path=robot_tag_path, revision='')
    if svn_revision(tag_svn_url):
        logging.info('Tag %s already exists.' % tag_svn_url)
        return tag_svn_url
    Arcadia.create_tag(svn_url, tag_svn_url, 'zomb-sandbox-rw', parents=True)
    return tag_svn_url


def get_real_urls(url_tpl, num=1):
    from api.cms import Yr

    # TODO: Document the function
    # TODO: Use CMS API to phconfig/host.cfg instead of yr (INFRA-443)
    rnd_urls = []
    search_res = re.search(r'(?P<tpl><(?P<yr_group>[^<>]+)>)', url_tpl)
    if not search_res:
        rnd_urls.append(url_tpl)
        return rnd_urls
    tpl = search_res.group('tpl')
    yr_group = search_res.group('yr_group')
    yr_list = Yr.listServersRaw('+%s' % yr_group)
    num = min(num, len(yr_list))
    rnd = random.Random()
    for rnd_fqdn in rnd.sample(yr_list, num):
        rnd_short_hostname = rnd_fqdn.split('.', 1)[0]
        rnd_url = url_tpl.replace(tpl, rnd_short_hostname)
        rnd_urls.append(rnd_url)
    logging.info('Random URL(s): %s.' % rnd_urls)
    return rnd_urls


def runner(thread_cb, tfunc, *tfunc_args, **tfunc_kwargs):
    # TODO: Document the function
    try:
        thread_cb['result'] = tfunc(*tfunc_args, **tfunc_kwargs)
    except Exception as exc:
        thread_cb['result'] = exc
        thread_cb['exception_str'] = traceback.format_exc()


def wait_for_threads(thread_cbs, join_timeout=5.0):
    """
        Try to join threads from @thread_cbs every @join_timeout seconds. For
        every finished thread check whether it finished successfully or by
        exception and reraise the exception in the latter case.
        @return: empty @thread_cbs dictionary.
    """
    logging.info('Waiting for threads: %s.' % thread_cbs.keys())
    while thread_cbs:
        for mode, thread_cb in thread_cbs.items():
            thread_cb['thread'].join(timeout=join_timeout)
            if not thread_cb['thread'].is_alive():
                logging.info('Thread %s finished.' % mode)
                if isinstance(thread_cb['result'], Exception):
                    raise TaskFail(
                        'Thread %s failed:\n%s' % (mode, thread_cb['exception_str'])
                    )
                del thread_cbs[mode]
    logging.info('Threads finished.')
    return thread_cbs


def repeat_func(func, iter_val, val_list, *args, **kwargs):
    # TODO: Document the function
    var = None
    for key, val in kwargs.items():
        if val == iter_val:
            var = key
            break
    if not var:
        raise TaskFail(
            'None of %s arguments is set to %s.' % (func.__name__, iter_val)
        )
    for val in val_list:
        try:
            kwargs[var] = val
            func(*args, **kwargs)
            return
        except Exception, e:
            logging.exception(e)
    raise TaskFail(
        '%s failed %d times for: %s.' % (func.__name__, len(val_list), val_list)
    )


# vim:set ts=4 sw=4 et:
