# -*- coding: utf-8 -*-

import __builtin__

import os

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects import lemur as new_resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.BaseSimpleBuildTaskWithPkg import BaseSimpleBuildTaskWithPkg
from sandbox.projects.common.arcadia import sdk

import sandbox.projects.common.build.parameters as build_params


def target2field_name(target):
    return 'build_%s' % target


class BuildLemur(BaseSimpleBuildTaskWithPkg):
    type = 'BUILD_LEMUR'
    client_tags = ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE

    TARGET_RESOURCES = (
        (new_resource_types.LEMUR_EXPORTPROCESSOR, 'robot/lemur/exportprocessor/main/exportprocessor'),
        (new_resource_types.LEMUR_FACTORSPROCESSOR, 'robot/lemur/factorsprocessor/main/factorsprocessor'),
        (new_resource_types.LEMUR_HARVESTER, 'robot/lemur/harvester/main/harvester'),
        (new_resource_types.LEMUR_LEMURWORM, 'robot/lemur/tools/lemurworm/lemurworm'),
        (new_resource_types.LEMUR_MASSREDIR_MERGER, 'robot/lemur/tools/massredir_merger/massredir_merger'),
        (resource_types.LEMUR_MRAGENT, 'robot/lemur/tools/lemur_mragent/lemur_mragent'),
        (new_resource_types.LEMUR_PREPARATPROCESSOR, 'robot/lemur/preparatprocessor/main/preparatprocessor'),
        (new_resource_types.LEMUR_PREPARE_GEMINI, 'robot/lemur/tools/prepare_gemini/prepare_gemini'),
        (new_resource_types.LEMUR_PREPARE_OWNER_DATA, 'robot/lemur/tools/prepare_owner_data/prepare_owner_data'),
        (new_resource_types.LEMUR_SENDLINKPROCESSOR, 'robot/lemur/sendlinkprocessor/main/sendlinkprocessor'),
        (new_resource_types.LEMUR_SRTOOL, 'robot/selectionrank/srtool/srtool'),
        (new_resource_types.LEMUR_UPLOAD_CANONIZATION_DATA, 'robot/lemur/tools/upload_canonization_data/upload_canonization_data'),
        (new_resource_types.LEMUR_UPLOAD_RENEWED_DATA, 'robot/lemur/tools/upload_renewed_data/upload_renewed_data'),
        (new_resource_types.LEMUR_VINTAGE, 'robot/lemur/vintage/main/vintage'),
        (new_resource_types.LEMUR_COUNTERS_TOOL, 'robot/lemur/tools/counters_tool'),
        (new_resource_types.LEMUR_QUEUEPROCESSOR, 'robot/lemur/queueprocessor/main/queueprocessor'),
        (new_resource_types.LEMUR_CRAWL_METRICS, 'robot/lemur/tools/crawl_metrics/main/crawl_metrics'),
        (new_resource_types.LEMUR_WEBMASTER_DATA, 'robot/lemur/tools/external_sources/webmaster-data'),
        (new_resource_types.LEMUR_NIRVANA_GRAPH_STARTER, 'yweb/robot/ukrop/nirvana_tools/graph_starter/graph-starter'),
        (resource_types.UKROP_FACTORS_HANDLER, 'yweb/robot/ukrop/tools/factors_handler/factors_handler'),
        (new_resource_types.LEMUR_RANKS_STAT, 'robot/lemur/tools/ranks_stat/ranks_stat'),
        (new_resource_types.LEMUR_HOST_SAMPLE_LIMITER, 'robot/lemur/tools/host_sample_limiter/host_sample_limiter'),
        (resource_types.LEMUR_SELECTION_RANK_LEARN, 'robot/selectionrank/sr_conduct_rules_experiment/sr_conduct_rules_experiment'),
        (new_resource_types.LEMUR_SPLIT_RENEWED_DATA, 'robot/lemur/tools/split_renewed_data/split_renewed_data'),
        (new_resource_types.LEMUR_EXTRACT_TEST_DATA, 'robot/lemur/tools/extract_test_data/extract_test_data'),
        (resource_types.UKROP_PRINTHISTORECS, 'yweb/robot/ukrop/tools/printhistorecs/printhistorecs'),
        (new_resource_types.LEMUR_ZORACL, 'yweb/robot/zoracl/main/zoracl'),
        (new_resource_types.LEMUR_ACTUALIZE_KIWI_EXPORT_MERGER, 'yweb/robot/ukrop/tools/actualize_kiwi_export_merger/actualize_kiwi_export_merger'),
        (new_resource_types.LEMUR_DYNAMIC_URLIFNO_UPDATER, 'yweb/robot/metrics/pmusca/update_lemur_urlinfo/bin'),
        (new_resource_types.LEMUR_UPDATE_LEMUR_URL_DATA, 'robot/lemur/tools/update_lemur_url_data'),
        (new_resource_types.LEMUR_UZOR_RESPONSE_PROCESSOR, 'robot/lemur/uzor_response_processor'),
        (new_resource_types.LEMUR_SEND_OVS_HOST_STATS_TO_SOLOMON, 'robot/lemur/tools/send_ovs_host_stats_to_solomon'),
        (new_resource_types.LEMUR_COUNT_CANDIDATES_TOP, 'robot/lemur/tools/count_candidates_top'),
    )

    PKG_CTX = {
        'repo_key_id': 'Ukrop Project (Yandex Crawler) <robot-ukrop-sandbox@yandex-team.ru>',
        'gpg_private_key': 'robot-ukrop-sandbox-gpg-private',
        'gpg_public_key': 'robot-ukrop-sandbox-gpg-public',
        'ssh_private_key': 'robot-ukrop-sandbox-ssh-private',
    }

    RESOURCE_TO_PKG_MAP = {
        'linux': {
            'LEMUR_EXPORTPROCESSOR': 'packages/debian/lemur/yandex-search-lemur-exportprocessor',
            'LEMUR_FACTORSPROCESSOR': 'packages/debian/lemur/yandex-search-lemur-factorsprocessor',
            'LEMUR_HARVESTER': 'packages/debian/lemur/yandex-search-lemur-harvester',
            'LEMUR_LEMURWORM': 'packages/debian/lemur/yandex-search-lemur-lemurworm',
            'LEMUR_MASSREDIR_MERGER': 'packages/debian/lemur/yandex-search-lemur-massredir-merger',
            'LEMUR_MRAGENT': 'packages/debian/lemur/yandex-search-lemur-mragent',
            'LEMUR_PREPARATPROCESSOR': 'packages/debian/lemur/yandex-search-lemur-preparatprocessor',
            'LEMUR_PREPARE_GEMINI': 'packages/debian/lemur/yandex-search-lemur-prepare-gemini',
            'LEMUR_PREPARE_OWNER_DATA': 'packages/debian/lemur/yandex-search-lemur-prepare-owner-data',
            'LEMUR_SENDLINKPROCESSOR': 'packages/debian/lemur/yandex-search-lemur-sendlinkprocessor',
            'LEMUR_SRTOOL': 'packages/debian/lemur/yandex-search-lemur-srtool',
            'LEMUR_UPLOAD_CANONIZATION_DATA': 'packages/debian/lemur/yandex-search-lemur-upload-canonization-data',
            'LEMUR_UPLOAD_RENEWED_DATA': 'packages/debian/lemur/yandex-search-lemur-upload-renewed-data',
            'LEMUR_VINTAGE': 'packages/debian/lemur/yandex-search-lemur-vintage',
            'LEMUR_COUNTERS_TOOL': 'packages/debian/lemur/yandex-search-lemur-counters-tool',
            'LEMUR_OUTLINKS_BLOOMFILTER': 'packages/debian/lemur/yandex-search-lemur-outlinks-bloomfilter',
            'LEMUR_QUEUEPROCESSOR': 'packages/debian/lemur/yandex-search-lemur-queueprocessor',
            'LEMUR_CRAWL_METRICS': 'packages/debian/lemur/yandex-search-lemur-crawl-metrics',
            'LEMUR_WEBMASTER_DATA': 'packages/debian/lemur/yandex-search-lemur-external-webmaster-data',
            'LEMUR_NIRVANA_GRAPH_STARTER': 'packages/debian/lemur/yandex-search-lemur-learn-ranks',
            'UKROP_FACTORS_HANDLER': 'packages/debian/ukrop/yandex-search-ukrop-factors-handler',
            'LEMUR_RANKS_STAT': 'packages/debian/lemur/yandex-search-lemur-learn-ranks',
            'LEMUR_HOST_SAMPLE_LIMITER': 'packages/debian/lemur/yandex-search-lemur-learn-ranks',
            'LEMUR_SELECTION_RANK_LEARN': 'packages/debian/lemur/yandex-search-lemur-learn-ranks',
            'LEMUR_SPLIT_RENEWED_DATA': 'packages/debian/lemur/yandex-search-lemur-split-renewed-data',
            'LEMUR_EXTRACT_TEST_DATA': 'packages/debian/lemur/yandex-search-lemur-extract-test-data',
            'UKROP_PRINTHISTORECS': 'packages/debian/ukrop/yandex-search-ukrop-printhistorecs',
            'LEMUR_ZORACL': 'packages/debian/lemur/yandex-search-lemur-zoracl',
            'LEMUR_ACTUALIZE_KIWI_EXPORT_MERGER': 'packages/debian/lemur/yandex-search-lemur-actualize-kiwi-export-merger',
            'LEMUR_DYNAMIC_URLIFNO_UPDATER': 'packages/debian/lemur/yandex-search-lemur-dynamic-urlinfo-update',
            'LEMUR_UPDATE_LEMUR_URL_DATA': 'packages/debian/lemur/yandex-search-lemur-update-lemur-url-data',
            'LEMUR_UZOR_RESPONSE_PROCESSOR': 'packages/debian/lemur/yandex-search-lemur-uzorresponseprocessor',
            'LEMUR_SEND_OVS_HOST_STATS_TO_SOLOMON': 'packages/debian/lemur/yandex-search-lemur-send-ovs-host-stats-to-solomon',
            'LEMUR_COUNT_CANDIDATES_TOP': 'packages/debian/lemur/yandex-search-lemur-count-candidates-top',
        },
    }

    class LemurBinaries(parameters.SandboxInfoParameter):
        name = 'info_field'
        description = 'Lemur binaries'

    input_parameters = BaseSimpleBuildTaskWithPkg.input_parameters + [build_params.StripBinaries] + [LemurBinaries]
    for _, path in TARGET_RESOURCES:
        target = os.path.basename(path)
        if path.endswith("/bin") or path.endswith("/server") or path.endswith("/main"):
            target = path.split('/')[-2]
        field_name = target2field_name(target)
        input_parameters.extend([__builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )])

    def initCtx(self):
        BaseSimpleBuildTaskWithPkg.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM
        self.ctx[build_params.StripBinaries.name] = True

    def get_definition_flags(self):
        default_defenition_flags = sdk.parse_flags(' '.join(['-DALLOCATOR=J',
                                                                    '-DCFLAGS=-fno-omit-frame-pointer',
                                                                    '-DROBOT_BUILD',
                                                                    '-DROBOT_OLDTOK']))
        if self.ctx.get('enable_sanitizer', False):
            sanitizer_flags = sdk.parse_flags(' '.join(['-DSANITIZER_TYPE=address',
                                                               '-DPIRE_DEBUG']))
            default_defenition_flags = dict(default_defenition_flags, **sanitizer_flags)
        external_definition_flags = sdk.parse_flags(self.ctx.get(consts.DEFINITION_FLAGS_KEY, ''))
        def_flags = dict(default_defenition_flags, **external_definition_flags)
        def_flags["SANDBOX_TASK_ID"] = self.id
        return def_flags

    def do_execute(self):
        if self.do_pkg_tasks_exist():
            self.check_pkg_tasks()
            return

        self.ctx['ya_yt_token_vault_owner'] = 'UKROP-ROBOT'
        self.ctx['ya_yt_token_vault_name'] = 'sandbox-build-yt-token'

        BaseSimpleBuildTaskWithPkg.do_execute(self)

        self.save_resources_ids('lemur.id.out.txt')

        if self.are_pkgs_enabled():
            self.mark_resources_ready()
            self.create_pkg_tasks(self.PKG_CTX)

    def save_resources_ids(self, ids_file):
        ids = []
        for resource_type, target, path in self.get_target_resources():
            field_name = target2field_name(target)
            if self.ctx[field_name]:
                id_key = '%s_resource_id' % target
                id_value = self.list_resources(resource_type)[0].id
                ids.append('"%s":"%s"' % (id_key, id_value))
        log_dir = get_logs_folder()
        with open(os.path.join(log_dir, ids_file), 'w') as ids_fd:
            ids_fd.write(','.join(ids))


__Task__ = BuildLemur
