# -*- coding: utf-8 -*-

import os
import tarfile
import contextlib
import platform

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk import paths
from sandbox.common import errors


class BuildLibOpenssl(ArcadiaTask):
    """
        Build openssl library
    """
    type = 'BUILD_LIBOPENSSL'

    execution_space = 30720

    BUILD_SYSTEM = consts.YMAKE_BUILD_SYSTEM
    TARGET_DIR = 'contrib/libs/openssl'
    LIBRARY_NAME = 'libopenssl'

    @staticmethod
    def parse_openssl_version(ver):
        # TODO: only recent version of openssl is supported
        major, minor, fix, patch_code, status_code = \
            ver >> 28, ver >> 20 & 0xff, ver >> 12 & 0xff, ver >> 4 & 0xff, ver & 0xf

        if patch_code:
            patch = chr(ord('a') + patch_code - 1)
        else:
            patch = ''

        if not status_code:
            status = '-dev'
        elif status_code < 0xe:
            status = '-beta{0}'.format(status_code)
        else:
            status = ''

        return '{major}.{minor}.{fix}{patch}{status}'.format(
            major=major, minor=minor, fix=fix, patch=patch, status=status)

    @classmethod
    def get_openssl_version(cls, path):
        with open(path, 'r') as fh:
            for line in fh:
                if not line.startswith('#'):
                    continue
                if 'OPENSSL_VERSION_NUMBER' not in line:
                    continue
                _, _, version = line.partition('OPENSSL_VERSION_NUMBER')
                version = version.strip()
                version = version[:-1] if version.endswith('L') else version
                return cls.parse_openssl_version(int(version.strip(), 16))
        return None

    def do_execute(self):
        build_system = self.BUILD_SYSTEM
        arcadia_src_dir = self.get_arcadia_src_dir()
        target_dirs = [self.TARGET_DIR]
        build_type = consts.RELEASE_BUILD_TYPE
        release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)
        def_flags = {'SANDBOX_TASK_ID': self.id}

        sdk.do_build(build_system, arcadia_src_dir, target_dirs, build_type, clear_build=True,
                            results_dir=release_dir, def_flags=def_flags)

        source_dir = os.path.join(arcadia_src_dir, self.TARGET_DIR)
        result_dir = self.abs_path(self.LIBRARY_NAME)
        include_result_dir = os.path.join(result_dir, 'include')
        library_result_dir = os.path.join(result_dir, 'lib')
        paths.make_folder(result_dir)

        paths.copy_path(
            os.path.join(source_dir, 'include'),
            include_result_dir,
            copy_function=os.link)
        paths.copy_path(
            os.path.join(release_dir, 'contrib', 'libs', 'openssl'),
            library_result_dir,
            copy_function=os.link)

        archive_path = '{}.tgz'.format(self.LIBRARY_NAME)
        with contextlib.closing(tarfile.open(archive_path, 'w:gz')) as tar_file:
            tar_file.add(result_dir, arcname=self.LIBRARY_NAME)

        version = self.get_openssl_version(os.path.join(source_dir, 'crypto', 'opensslv.h'))
        if version is None:
            raise errors.SandboxEnvironmentError('unable to detect openssl version')
        self.create_resource(
            description='{0} r{1}'.format(self.LIBRARY_NAME, utils.svn_revision(arcadia_src_dir)),
            resource_path=archive_path,
            resource_type=resource_types.OPENSSL_LIBRARY,
            attributes={'platform': platform.platform(), 'version': version},
            arch=self.arch)


__Task__ = BuildLibOpenssl
