# -*- coding: utf-8 -*-
import os
import shutil

import sandbox.common.types.client as ctc

from ConfigParser import ConfigParser

from sandbox.sandboxsdk import svn, parameters
from sandbox.sandboxsdk.paths import make_folder, list_dir
from sandbox.sandboxsdk.process import run_process

import sandbox.projects.common.build.parameters as build_params
from sandbox.projects import resource_types
from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask


class BuildMapreducelibPy(ArcadiaTask):
    """
        Сборка пакета с модулем mapreducelib.py
    """
    type = "BUILD_MAPREDUCELIB_PY"

    client_tags = ctc.Tag.Group.LINUX

    class UploadToPypi(parameters.SandboxBoolParameter):
        name = 'upload_to_pypi'
        description = 'Upload package to yandex pypi repository'
        default_value = False

    input_parameters = build_params.get_arcadia_params() + [UploadToPypi]

    secret_owner = "MR-USERS"
    secret_attr = "pypi_credentials"
    python_binary = "python2.7"
    package_source_arc_project = ["quality", "mapreducelib_py_distro"]
    module_source_arc_location = ["yweb", "scripts", "datascripts", "common", "mapreducelib.py"]

    @classmethod
    def _patch_config(cls, origConfigFile, patchedConfigFile, **kws):
        cp = ConfigParser()
        if origConfigFile not in cp.read(origConfigFile):
            raise RuntimeError("can't parse configuration file")
        for k, v in kws.iteritems():
            cp.set("global", k, v)
        with open(patchedConfigFile, "w") as patch_prn:
            cp.write(patch_prn)

    def _prepare_arcadia_src(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        svn.Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())
        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)
        return arcadia_src_dir

    def _prepare_pypirc(self, directory):
        if not os.path.isdir(directory):
            make_folder(directory)
        pypirc_data = self.get_vault_data(self.secret_owner, self.secret_attr)
        with open(os.path.join(directory, ".pypirc"), "w") as rc_prn:
            rc_prn.write(pypirc_data)

    def do_execute(self):
        arcadia_src_dir = self._prepare_arcadia_src()
        build_dir = self.path("mapreducelib_package")
        shutil.copytree(os.path.join(arcadia_src_dir, *self.package_source_arc_project), build_dir)
        shutil.copy(os.path.join(arcadia_src_dir, *self.module_source_arc_location), build_dir)
        distutl_config = os.path.join(build_dir, "setup.cfg")
        self._patch_config(distutl_config, distutl_config, version=self.ctx['arcadia_revision'])
        run_process(
            [self.python_binary, "setup.py", "bdist"],
            log_prefix="bdist creation",
            work_dir=build_dir
        )
        bdist_archive = list_dir(os.path.join(build_dir, "dist"), files_only=True, abs_path=True)[0]
        make_folder(os.path.join(build_dir, "pyshared"))
        run_process(
            ["tar", "xf", bdist_archive, "--strip-components", "6", "-C", os.path.join(build_dir, "pyshared")],
            log_prefix="bdist unarchive"
        )
        self.create_resource(
            description="distribution directory",
            resource_path=os.path.join(build_dir, "pyshared"),
            resource_type=resource_types.MAPREDUCELIBPPY_PACKAGE
        )
        if self.ctx[self.UploadToPypi.name]:
            self._prepare_pypirc(os.path.join(build_dir, "fake_home"))
            run_process(
                [self.python_binary, "setup.py", "sdist", "upload", "-r", "yandex"],
                log_prefix="sdist uploading",
                work_dir=build_dir,
                environment={"HOME": "fake_home"}
            )


__Task__ = BuildMapreducelibPy
