# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk.parameters import LastReleasedResource, ResourceSelector
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.BuildMapsFilteredHostFactors import BuildMapsFilteredHostFactors
from sandbox.projects.BuildMapsFilteredWebUserFactors import BuildMapsFilteredWebUserFactors
from sandbox.projects.common.geosearch.utils import unpack_file
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask
from sandbox.projects.common.utils import get_or_default, set_resource_attributes


POI_RUBRIC_ID = '184108083'


class FilterExecutableParameter(LastReleasedResource):
    name = 'filter'
    description = 'Filter executable'
    resource_type = resource_types.GEOBUSINESSDATAFILTER_EXECUTABLE


class BackaExportParameter(ResourceSelector):
    name = 'backa_export'
    description = 'Backa export archive'
    resource_type = resource_types.MAPS_DATABASE_BUSINESS_SOURCE_ARCHIVE
    required = True


class BuildMapsCompaniesPoiAndUrls(ReleaseToNannyTask, SandboxTask):
    """
        Build a filtered list of POI compaines and company urls.
    """
    type = 'BUILD_MAPS_COMPANIES_POI_AND_URLS'
    input_parameters = [FilterExecutableParameter, BackaExportParameter]

    def on_execute(self):
        filter_executable_path = self._get_filter_executable_path()
        companies2_xml_path = self._unpack_companies2_xml()

        output_dir = os.path.join(self.abs_path(), 'out')
        make_folder(output_dir)
        company_urls_path = os.path.join(output_dir, 'company_urls.json')
        poi_companies_path = os.path.join(output_dir, 'companies2.xml')

        cmd = [
            'python',
            filter_executable_path,
            companies2_xml_path,
            '--rubric_id', POI_RUBRIC_ID,
            '--out_company_urls', company_urls_path,
            '--out_filtered_companies', poi_companies_path,
        ]
        run_process(cmd, log_prefix='company_filter')

        company_urls = self.create_resource('Backa company urls',
                                            company_urls_path,
                                            resource_types.MAPS_DATABASE_BUSINESS_COMPANY_URLS)
        self.create_resource('Backa POI companies',
                             poi_companies_path,
                             resource_types.MAPS_DATABASE_BUSINESS_POI_COMPANIES)

        self.create_subtask(
            task_type='RELEASE_ANY',
            inherit_notifications=True,
            input_parameters={
                'release_task_id': self.id,
                'release_status': RELEASE_STABLE,
            },
            description=('backa POI companies and urls (task id: {}) autorelease' .format(self.id))
        )

        # Set finite ttl to the large released backa export archive.
        backa_export = self.ctx[BackaExportParameter.name]
        set_resource_attributes(backa_export, {'ttl': '14'})

        self.create_subtask(
            task_type=BuildMapsFilteredHostFactors.type,
            description='Filter host factors by new companies',
            input_parameters={
                'company_urls': company_urls.id,
            },
        )
        self.create_subtask(
            task_type=BuildMapsFilteredWebUserFactors.type,
            description='Filter user factors by new companies',
            input_parameters={
                'company_urls': company_urls.id,
            },
        )

    def _get_filter_executable_path(self):
        filter_executable = get_or_default(self.ctx, FilterExecutableParameter)
        filter_executable_dir = self.sync_resource(filter_executable)
        filter_executable_path = os.path.join(filter_executable_dir, 'companies_filter', 'main.py')
        return filter_executable_path

    def _unpack_companies2_xml(self):
        backa_export = self.ctx[BackaExportParameter.name]
        backa_export_archive = self.sync_resource(backa_export)
        backa_export_unpacked_dir = make_folder(os.path.join(self.abs_path(), 'backa_unpacked'))
        unpack_file(backa_export_archive, backa_export_unpacked_dir, arch_type='tar')
        companies2_xml_gz_path = os.path.join(backa_export_unpacked_dir, 'companies2.xml.gz')
        companies2_xml_path = os.path.join(backa_export_unpacked_dir, 'companies2.xml')
        unpack_file(companies2_xml_gz_path, companies2_xml_path)
        return companies2_xml_path


__Task__ = BuildMapsCompaniesPoiAndUrls
