# -*- coding: utf-8 -*-

import logging
import os

from sandbox.common.utils import md5sum
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.tycoon.TycoonAdverts import TycoonAds
from sandbox.projects.common.geosearch.indexing import BuildHelper
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask
from sandbox.projects.common.utils import get_or_default


class DataSourceParameter(ResourceSelector):
    name = 'data_source_resource_id'
    description = 'Data source'
    resource_type = TycoonAds
    group = 'Maps Indexer Advert parameters'


class AdvertIndexerParameter(LastReleasedResource):
    name = 'advert_indexer_resource_id'
    description = 'Indexer executable: '
    resource_type = resource_types.GEOADVERTINDEXER_EXECUTABLE
    group = 'Maps Indexer Advert parameters'


def filepath(dir_, name):
    path = os.path.join(dir_, name)
    if not os.path.isfile(path):
        raise SandboxTaskFailureError('data source directory must contain ' + name)
    return path


class BuildMapsDatabaseAdvert(ReleaseToNannyTask, SandboxTask):
    """
        Build maps database advert
    """
    type = 'BUILD_MAPS_DATABASE_ADVERT'

    input_parameters = (
        DataSourceParameter,
        AdvertIndexerParameter
    )
    cores = 1

    def sync_advert_source(self, source):
        resource_id = get_or_default(self.ctx, source)
        if not resource_id:
            raise SandboxTaskFailureError('data source is required')
        logging.info('Got source %s' % source.name)

        source_path = self.sync_resource(resource_id)
        if os.path.isfile(source_path):
            logging.info('treating source %s as xml only with no images' % source.name)
            return source_path

        if not os.path.isdir(source_path):
            raise SandboxTaskFailureError('data source must be either file or directory (%s)' % source.name)

        ads_xml = filepath(source_path, 'ads.xml')
        return ads_xml

    def build_main_source(self, source, index_directory):
        source_xml = self.sync_advert_source(source)

        logging.info('Building pb...')
        run_process([self.indexer, '--output-dir', index_directory, source_xml],
                    log_prefix='indexer, ads.xml to pb')

    def sync_indexer(self):
        self.indexer = self.sync_resource(self.ctx[AdvertIndexerParameter.name])

    def on_execute(self):
        self.sync_indexer()
        index_directory = 'index'

        make_folder(index_directory)

        self.build_main_source(DataSourceParameter, index_directory='index')

        attrs = {}
        file_to_hash = os.path.join(index_directory, 'menu_advert.pb.bin')
        if os.path.isfile(file_to_hash):
            attrs['md5(menu_advert.pb.bin)'] = md5sum(file_to_hash)

        # publish without tar
        BuildHelper.create_index_resource(self,
                                          primary_source_resource_id=self.ctx[DataSourceParameter.name],
                                          index_directory=index_directory,
                                          index_type=resource_types.MAPS_DATABASE_ADVERT,
                                          additional_attributes=attrs)

    def on_release(self, parameters):
        self.mark_released_resources(parameters["release_status"], ttl=60)


__Task__ = BuildMapsDatabaseAdvert
